#
# 1. Save what valid positions we have
# 2. Have a flag to indicate if dec clamp is in place
#
# $Name:  $
#
# $Log: Control.pm,v $
# Revision 0.5  2005/01/16 03:21:09  robert
# Switch from Time::Object to Date::Time
# Add option to ignore maservo errors
# Add FITS header items observat, observer, telescop, equinox,
# DARKTIME, CTYPE, CRPIX, CD, CRVAL
# Change FITS date_obs to date-obs
# Add FITS file suffix support
# Add chiller support
# Add flag for DEC clamp usage
#
# Revision 0.4  2003/01/25 19:15:20  robert
# Add more parameter control to ini file
# Genericize routines to share more code
# Add preliminary support for traking motor
#   positions when software isn't running
#
# Revision 0.3  2002/12/23 04:18:39  robert
# First pass of documentation for all methods.
# Modify image download to use one persistent child process and socket
#   communication.
# Change config file DAC start/end settings to be more friendly.
# Keep track of some extra statistics (image counts).
# Modify ra_check_position to drive Ra to closest limit if either limit is
#   excedded and return status rather than dying.
# Fix clear_ccds method.
#
# Revision 0.2  2002/11/16 04:48:12  robert
# Add config file support.  Start documentation.
#
# Revision 0.1.1.1  2002/11/03 02:53:17  robert
#
#
package TASS::Control;

use 5.006;
use strict;
use warnings;

use TASS::Control::Driver;
use Log::Agent;
use Log::Agent::Priorities qw( :LEVELS );
use Log::Agent::Tag::Callback;
use Time::HiRes qw( time gettimeofday sleep tv_interval );
use Astro::Time;
use Proc::Fork;
use POSIX ":sys_wait_h";
use Config::IniFiles;
use DateTime;
use IO::Socket;
use IO::Select;

my $ID = q$Id: Control.pm,v 0.5 2005/01/16 03:21:09 robert Exp $;
my $version = join( ' ', ( split ( ' ', $ID ) )[ 1 .. 3 ] );
$version =~ s/,v\b//;
$version =~ s/(\S+)$/($1)/;
our $VERSION = $version;

$SIG{CHLD} = 'IGNORE';
$SIG{INT} = sub { exit; };
$SIG{TERM} = sub { exit; };

=begin testing

use Log::Agent;
use Log::Agent::Priorities qw( :LEVELS );
require Log::Agent::Driver::File;
require Log::Agent::Tag::Callback;

BEGIN{ use_ok( 'TASS::Control' ); }

my $no_hash = TASS::Control->new;
ok( defined $no_hash, 'defined from new with no hash' );
isa_ok( $no_hash, 'TASS::Control', 'ISA TASS::Control' );
#
# Set up the logging
unlink 'td.err', 'td.dbg', 'td.out';
logconfig( -driver => Log::Agent::Driver::File->make(
                         -prefix => 'TC',
                         -showpid => 1,
                         -channels => { 'error' => 'td.err',
                                        'output' => 'td.out',
                                        'debug' => 'td.dbg' } ),
           -level => INFO
         );

my $hash = TASS::Control->new( undef, ( 'port' => '/dev/ttyS0',
                                        'baud' => '9600' ) );

#$hash->start;

=end testing

=head1 NAME

TASS::Control - Controls a TASS Mark IV telescope.

=head1 SYNOPSIS

=for example begin

 use TASS::Control;

 $ts = TASS::Control->new;

=for example end

=head1 DESCRIPTION

Provides a nice interface to control the telescope through
TASS::Control::Driver.  To ease use, any problems encountered from
TASS::Control::Driver calls will cause this module to attempt to shut
down all motors and then exit.  Retries on all commands are handled through
TASS::Control::Driver.  Although restrictive sounding, I've not had many
problems of premature exit through 50+ sunset to sunrise runs.

In order to provide orderly shutdown (and not leave the RA drive on),
both INT and TERM signals are caught and directed through exit to call
the object destructor, which will shutdown down all motors.

This module forks one process to handle all picture downloads, so it
also captures the CHLD signal and sets it to 'IGNORE' in case the
parent manages to exit without telling the child to quit.  This prevents
zombie processes from accumulating.

=head1 METHODS

=over

=item B<New>

=over

=item C<OBJREF = TASS::Control-E<gt>new>

=item C<OBJREF = TASS::Control-E<gt>new( SCALAR )>

=item C<OBJREF = TASS::Control-E<gt>new( SCALAR, HASH )>

Creates a Tass Control object and returns a reference to the object.
Either takes nothing, or an initialization file name and hash of
L<attributes|"ATTRIBUTES">.  If the file name is not defined (first form or
third form with 'undef' as the file name), the file 'tass-control.cfg' will be
created with default values based on ROB.  The same will occur if a file
name is given but that file doesn't exist, it will be created with default
values.  Valid values within the hash will override the configuration file
parameters.

=item I<For Example>

=for example begin

 $tc = TASS::Control->new;
 $tc = TASS::Control->new( 'my_config.file' );
 $tc = TASS::Control->new( undef, ( 'port' => '/dev/ttyS1' ) );

=for example end

=back

=back

=cut

use Class::MethodMaker
   new_with_init => 'new',
   new_hash_init => '_hash_init',
   #
   # Stuff
   get_set       => [ qw/_socket/ ],
   boolean       => [ qw/_in_progress _child/ ],
   #
   # To send to TASS::Control::Driver, and the driver itself
   get_set       => [ qw/port baud driver ignore_maservo_errors/ ],
   #
   # Site information
   get_set       => [ qw/longitude latitude site_code/ ],
   get_set       => [ qw/observat observer telescop/ ],
   #
   # Transformed position information
   get_set       => [ qw/_t_longitude/ ],
   #
   # Generic stuff
   get_set       => [ qw/position_file/ ],
   boolean       => [ qw/_driving_to_limit/ ],
   #
   # Chiller information
   boolean       => [ qw/chiller_use/ ],
   get_set       => [ qw/max_time_on min_time/ ],
   get_set       => [ qw/duty_cycle _chiller_last_time/ ],
   get_set       => [ qw/delta_temp min_temp dead_band/ ],
   get_set       => [ qw/_chiller_current_run_time/ ],
   get_set       => [ qw/_chiller_time_on _chiller_time_off/ ],
   boolean       => [ qw/_chiller_on/ ],
   #
   # Focus information
   get_set       => [ qw/focus_0_position focus_1_position/ ],
   get_set       => [ qw/focus_0_best focus_1_best/ ],
   get_set       => [ qw/focus_0_range focus_1_range/ ],
   get_set       => [ qw/focus_0_last_direction focus_1_last_direction/ ],
   boolean       => [ qw/focus_0_motor_on focus_1_motor_on/ ],
   #
   # Dec drive information
   get_set       => [ qw/dec_drive_position dec_drive_steps_degree/ ],
   get_set       => [ qw/dec_drive_level_from_limit/ ],
   get_set       => [ qw/dec_drive_range/ ],
   boolean       => [ qw/dec_drive_motor_on/ ],
   get_set       => [ qw/dec_drive_backlash dec_drive_last_direction/ ],
   #
   # Dec clamp information
    boolean      => [ qw/dec_clamp_use/ ],
   get_set       => [ qw/dec_clamp_position dec_clamp_in dec_clamp_out/ ],
   #
   # Shutter information
   get_set       => [ qw/shutter_0_close_pos shutter_0_open_pos/ ],
   get_set       => [ qw/shutter_1_close_pos shutter_1_open_pos/ ],
   get_set       => [ qw/camera_0_filter camera_1_filter/ ],
   #
   # RA information
   get_set       => [ qw/ra_position/ ],
   get_set       => [ qw/ra_level_from_limit/ ],
   get_set       => [ qw/ra_range/ ],
   get_set       => [ qw/ra_speed ra_time_start ra_max_speed/ ],
   #
   # DAC info
   get_set       => [ qw/dac_00_start dac_00_end/ ],
   get_set       => [ qw/dac_01_start dac_01_end/ ],
   get_set       => [ qw/dac_02_start dac_02_end/ ],
   get_set       => [ qw/dac_03_start dac_03_end/ ],
   get_set       => [ qw/dac_04_start dac_04_end/ ],
   get_set       => [ qw/dac_05_start dac_05_end/ ],
   get_set       => [ qw/dac_06_start dac_06_end/ ],
   get_set       => [ qw/dac_07_start dac_07_end/ ],
   get_set       => [ qw/dac_08_start dac_08_end/ ],
   get_set       => [ qw/dac_09_start dac_09_end/ ],
   get_set       => [ qw/dac_10_start dac_10_end/ ],
   get_set       => [ qw/dac_11_start dac_11_end/ ],
   get_set       => [ qw/dac_12_start dac_12_end/ ],
   get_set       => [ qw/dac_13_start dac_13_end/ ],
   get_set       => [ qw/dac_14_start dac_14_end/ ],
   get_set       => [ qw/dac_15_start dac_15_end/ ],
   #
   # Picture information
   hash          => [ qw/pic_info_num pic_info_str pic_info_comments/ ],
   get_set       => [ qw/_picture_time _last_picture_end/ ],
   get_set       => [ qw/naxis1 naxis2/ ],
   get_set       => [ qw/dark_diff max_object_dark_time/ ],
   counter       => [ qw/_bias_count _dark_count _object_count/ ],
   counter       => [ qw/_bad_transfer _overexposure/ ],
   get_set       => [ qw/equinox file_suffix/ ];

sub _setup_var
   {
   my $self = shift;
   my( $ini, $section, $parameter, $default, $comment ) = @_;

   #
   # When the section/parameter doesn't exist
   if ( !exists $ini->{$section}{$parameter} )
      {
      #
      # We'll set up with the default
      $ini->{$section} = { } if !exists $ini->{$section};
      $ini->{$section}{$parameter} = $default;
      #
      # Add add our comment
      if (  defined( $comment )
         && ref( $comment ) )
         {
         #
         # An array of comments
         (tied %$ini)->SetParameterComment( $section, $parameter, @$comment );
         }
      else
         {
         #
         # A potential single comment
         (tied %$ini)->SetParameterComment( $section, $parameter, $comment )
            if defined $comment;
         }
      }
   #
   # And finally, setup the parameter to use internally
   $self->$parameter( $ini->{$section}{$parameter} );
   }

sub init
   {
   my $self = shift;
   my $ini_file = shift || 'tass-control.cfg';
   my %args = @_;
   my %ini;
   my @settings;


   if ( -e $ini_file )
      {
      tie %ini, 'Config::IniFiles', ( -file => $ini_file );
      }
   else
      {
      tie %ini, 'Config::IniFiles';
      (tied %ini)->SetFileName( $ini_file );
      }

   #
   # Comments for all items going into the FITS header
   $self->pic_info_comments(
      'date-obs', 'Universal time of of mid exposure',
      'adc_00', 'Water Temp 0',
      'adc_01', 'VCO Temp',
      'adc_02', 'VCO Reference',
      'adc_03', 'Temp 1',
      'adc_04', 'Temp 2',
      'adc_05', 'Temp 3',
      'adc_06', 'Temp C0 DAC15',
      'adc_07', 'Temp C3 DAC10',
      'adc_08', 'Ground',
      'adc_09', '-15',
      'adc_10', '-5',
      'adc_11', '+5',
      'adc_12', '+15',
      'adc_13', 'VCO DAC12',
      'adc_14', 'Temp C1 DAC14',
      'adc_15', 'Temp C2 DAC11',
      'adc_16', 'CCD Temp 2',
      'adc_17', 'CCD VDD Power',
      'adc_18', 'CCD Temp 0',
      'adc_19', 'CCD Temp 1',
      'adc_20', 'ADC Reference',
      'adc_21', 'VVH DAC 7',
      'adc_22', 'VRH DAC 2',
      'adc_23', 'VHL DAC 1',
      'adc_24', 'TEC Current 2',
      'adc_25', 'TEC Current 0',
      'adc_26', 'TEC Current 1',
      'adc_27', 'TEC Current 3',
      'adc_28', 'TEC Power',
      'adc_29', 'VVL DAC 6',
      'adc_30', 'VRL DAC 3',
      'adc_31', 'VHH DAC 0',
      'DARKTIME', 'Amount of extra shutter close time in the exposure',
      'CTYPE1', 'Gnomonic projection X axis',
      'CTYPE2', 'Gnomonic projection Y axis',
      'CRPIX1', 'Pixel coordinate of reference point X axis',
      'CRPIX2', 'Pixel coordinate of reference point Y axis',
      'CD1_1',  'Conversion matrix element',
      'CD2_2',  'Conversion matrix element',
      'CRVAL1', 'Galactic longitude at reference point X axis',
      'CRVAL2', 'Galactic latitude at reference point Y axis'
      );

   #
   # Misc stuff
   $self->_bad_transfer_reset;
   $self->_overexposure_reset;
   $self->_bias_count_reset;
   $self->_dark_count_reset;
   $self->_object_count_reset;
   $self->clear__in_progress;
   $self->clear__child;
   $self->clear__driving_to_limit;
   #
   # Just to make them defined
   $self->_last_picture_end( [0, 0] );
   $self->_chiller_last_time( [gettimeofday] );
   $self->_chiller_time_on( 0 );
   $self->_chiller_time_off( 0 );
   $self->_chiller_current_run_time( 0 );

   #
   # We don't know where we are or what the motors are doing
   $self->clear_dec_drive_position;
   $self->clear_dec_clamp_position;
   $self->clear_ra_position;
   #
   # set up some defaults based on ROB
   push @settings,
      [ qw:computer port /dev/ttyS0:, 'Serial port' ],
      [ qw/computer baud 9600/ ],

      [ qw/misc ignore_maservo_errors 0/, 'Ignores maservo errors if 1' ],

      [ qw/site site_code k/, 'Single lowercase character' ],
      [ qw/site longitude/, str2deg( '-105:10:50', 'D' ), 'Decimal degrees' ],
      [ qw/site latitude/, str2deg( '40:19:25', 'D' ), 'Decimal degrees' ],
      [ qw/site observat/, 'Berthoud, CO', 'Observatory location' ],
      [ qw/site observer/, 'Robert Creager', 'Observer name location' ],
      [ qw/site telescop/, 'ROB', 'Telescope name' ],

      [ qw/chiller chiller_use 1/, '1/0 if using a chiller or not' ],
      [ qw/chiller duty_cycle 50/,
        'Max time chiller will be run, decimal percent' ],
      [ qw/chiller max_time_on 900/,
        'Maximum time chiller will be on or off at a time in seconds' ],
      [ qw/chiller min_time 300/,
        'Minimum time chiller will be on or off at a time in seconds' ],
      [ qw/chiller delta_temp 15/,
        'The desired air/water temperature delta' ],
      [ qw/chiller min_temp 5/,
        'The minimum temperature to cool the water to' ],
      [ qw/chiller dead_band 2/,
        'The chiller controller dead band' ],

      [ qw/camera naxis1 2064/ ],
      [ qw/camera naxis2 2037/ ],
      [ qw/camera file_suffix .fits/,
          'The desired suffix for the FITS files' ],
      [ qw/camera dark_diff 10/,
        [ 'Subtracted from the dark exposure time to determine the',
          'maximum time lapsed from the last exposre to not execute',
          'a clearing scan.  In other words, how accurate the dark exposure',
          'is on the + side only.  The number can be less than 0, and if',
          'it is less than about 3, the dark exposure may be long' ], ],
      [ qw/camera max_object_dark_time 1/,
        [ 'The maximum extra shutter close time from the last picture end',
          'before a clearing scan is executed.  Only affects the dark time',
          'of the first exposure in a set, not the shutter open time' ], ],
      [ qw/camera camera_0_filter V/, 'Filter for camera 0' ],
      [ qw/camera camera_1_filter I/, 'Filter for camera 1' ],
      [ qw/camera focus_0_range 1500/, 'Focus range for camera 0' ],
      [ qw/camera focus_1_range 1300/, 'Focus range for camera 0' ],
      [ qw/camera focus_0_best 550/, 'Best focus position for camera 0' ],
      [ qw/camera focus_1_best 300/, 'Best focus position for camera 1' ],
      [ qw/camera shutter_0_open_pos 220/, 'Absolute servo position' ],
      [ qw/camera shutter_0_close_pos 40/, 'Absolute servo position' ],
      [ qw/camera shutter_1_open_pos 230/, 'Absolute servo position' ],
      [ qw/camera shutter_1_close_pos 55/, 'Absolute servo position' ],

      [ qw/telescope position_file position_file.ini/,
        'Stores the positions of the motors after program exit' ],
      [ qw/telescope dec_drive_steps_degree/, 1627 / 13.5 ],
      [ qw/telescope dec_drive_level_from_limit 150/,
        'Steps from limit, optics horizontal' ],
      [ qw/telescope dec_drive_range 20170/, 'Steps' ],
      [ qw/telescope dec_drive_backlash 1/, 'Steps' ],
      [ qw/telescope dec_clamp_use 0/, 'Whether to use clamp or not' ],
      [ qw/telescope dec_clamp_in 400/, 'Steps' ],
      [ qw/telescope dec_clamp_out 400/, 'Steps' ],
      [ qw/telescope ra_level_from_limit 3438/,
        'Siderial seconds from meridian to limit' ],
      [ qw/telescope ra_max_speed 32/,
        'Maximum speed to drive RA at - must be power of 2' ],

      [ qw/dac dac_00_start 6/, 'VHH - high horizontal CCD clock level' ],
      [ qw/dac dac_01_start -3/, 'VHL - low horizontal CCD clock level' ],
      [ qw/dac dac_02_start 0/, 'VRH - high reset CCD clock level (> VRH)' ],
      [ qw/dac dac_03_start 7/, 'VRL - low reset CCD clock level' ],
      [ qw/dac dac_04_start 2/,
           'TR ADC3 - trim pedestal voltage for ADC channel 3' ],
      [ qw/dac dac_05_start 3/,
           'TR ADC2 - Trim pedestal voltage for ADC channel 2' ],
      [ qw/dac dac_06_start -8/, 'VVL - low verticle CCD clock level' ],
      [ qw/dac dac_07_start 3.9/, 'VVH - high vertical CCD clock level' ],
      [ qw/dac dac_08_start 0/,
           'TR ADC1 - trim pedestal voltage for ADC channel 1 (camera 1)' ],
      [ qw/dac dac_09_start 0/,
           'TR ADC0 - trim pedestal voltage for ADC channel 0 (camera 0)' ],
      [ qw/dac dac_10_start 6/, 'T COM3' ],
      [ qw/dac dac_11_start 7/, 'T COM2' ],
      [ qw/dac dac_12_start .2/, 'VCO - trims VCO value about 2%' ],
      [ qw/dac dac_13_start 9/,
           'Spare - comes out on the auxiliary connector' ],
      [ qw/dac dac_14_start 2/, 'T COM1 - temperature for camera 1' ],
      [ qw/dac dac_15_start 2/, 'T COM0 - temperature for camera 0' ],
      [ qw/dac dac_00_end 6/, 'VHH - high horizontal CCD clock level' ],
      [ qw/dac dac_01_end -3/, 'VHL - low horizontal CCD clock level' ],
      [ qw/dac dac_02_end 0/, 'VRH - high reset CCD clock level (> VRH)' ],
      [ qw/dac dac_03_end 7/, 'VRL - low reset CCD clock level' ],
      [ qw/dac dac_04_end 2/,
           'TR ADC3 - trim pedestal voltage for ADC channel 3' ],
      [ qw/dac dac_05_end 3/,
           'TR ADC2 - Trim pedestal voltage for ADC channel 2' ],
      [ qw/dac dac_06_end -8/, 'VVL - low verticle CCD clock level' ],
      [ qw/dac dac_07_end 3.9/, 'VVH - high vertical CCD clock level' ],
      [ qw/dac dac_08_end 0/,
           'TR ADC1 - trim pedestal voltage for ADC channel 1 (camera 1)' ],
      [ qw/dac dac_09_end 0/,
           'TR ADC0 - trim pedestal voltage for ADC channel 0 (camera 0)' ],
      [ qw/dac dac_10_end 6/, 'T COM3' ],
      [ qw/dac dac_11_end 7/, 'T COM2' ],
      [ qw/dac dac_12_end .2/, 'VCO - trims VCO value about 2%' ],
      [ qw/dac dac_13_end 9/, 'Spare - comes out on the auxiliary connector' ],
      [ qw/dac dac_14_end 10/, 'T COM1 - temperature for camera 1' ],
      [ qw/dac dac_15_end 10/, 'T COM0 - temperature for camera 0' ];

   foreach ( @settings )
      {
      $self->_setup_var( \%ini, @$_ );
      }

   #
   # In seconds
   $self->ra_range( $self->ra_level_from_limit * 2 );

   $self->_hash_init( %args );
   #
   # And some derived stuff
   $self->_t_longitude( deg2turn( $self->longitude ) );

   #
   # These don't change
   $self->equinox( 2000 );

   $self->pic_info_num( 'longitud', $self->longitude );
   $self->pic_info_num( 'latitude', $self->latitude );
   $self->pic_info_str( 'observat', $self->observat );
   $self->pic_info_str( 'observer', $self->observer );
   $self->pic_info_str( 'telescop', $self->telescop );
   $self->pic_info_str( 'CTYPE1', 'DEC--TAN' );
   $self->pic_info_str( 'CTYPE2', 'RA---TAN' );
   $self->pic_info_num( 'CRPIX1', $self->naxis1 / 2 );
   $self->pic_info_num( 'CRPIX2', $self->naxis2 / 2 );
   $self->pic_info_num( 'CD1_1',  0.002126 );
   $self->pic_info_num( 'CD2_2', -0.002126 );
   $self->pic_info_num( 'equinox', $self->equinox );

   (tied %ini)->RewriteConfig;
   untie %ini;
   #
   # Load any possibly known positions
   $self->_load_positions;
   }

=over

=item B<Start>

=over

=item C<$tc-E<gt>start>

Instantiates the internal TASS::Control::Driver object with the configured port
and baud.  No other methods can be used without first executing this method.
Additionally, loads the 'MPX' register with 0.

=item I<For Example>

=for example begin

 $tc->start;

=for example end

=back

=back

=cut

sub start
   {
   my $self = shift;

   my $fh_parent = IO::Handle->new;
   my $fh_child = IO::Handle->new;
   socketpair( $fh_child, $fh_parent, AF_UNIX, SOCK_STREAM, PF_UNSPEC )
      or die "socketpair: $!";

   $fh_parent->autoflush( 1 );
   $fh_child->autoflush( 1 );

   my $driver;

   parent
      {
      $self->_socket( $fh_parent );
      $fh_child->close;
      $driver = TASS::Control::Driver->new( ( 'port' => $self->port,
                                              'baud' => $self->baud,
                                              'ignore_maservo_errors'
                                               => $self->ignore_maservo_errors,
#                                              'simulate' => 'false'
                                            ) );
      $self->driver( $driver );
      #
      # Die, even though the driver should also on failure
      die 'Error: Could not start up the driver' if $self->driver->start;
      #
      # Set up the correct eprom to use
      $self->driver->load_register( 'MPX', 0 );
      #
      # Reset the memory card per Service note 4
      $self->_outb( '0x300', '0x80', '0x05', '0x00' );
      $self->driver->pulse( 17 );
      }
   child
      {
      $self->_socket( $fh_child );
      $self->set__child;
      $fh_parent->close;

      $self->_download_server;

      exit 0;
      }
   error
      {
      die 'Yikes - real problem during the start';
      };
   }

=over

=item B<Initialize Telescope>

=over

=item C<$tc-E<gt>init_telescope>

Initializes the telescope for use.  Turns off all motors, closes the dec
clamp, drives the dec motor to it's limit, drives the ra motor to it's limit,
closes the camera shutters.

=item I<For Example>

=for example begin

 $tc->init_telescope;

=for example end

=back

=back

=cut

sub init_telescope
   {
   my $self = shift;
   #
   # Make sure all the motors that are on will be off
   $self->all_off;
   #
   # If we don't know where we are for dec clamp, drive it closed
   if ( !defined $self->dec_clamp_position )
      {
      $self->dec_clamp_position( $self->dec_clamp_out );
      if ( $self->dec_clamp_use )
         {
         $self->dec_clamp_close;
         }
      }
   #
   # If we don't know where we are for ra drive, drive it home
   $self->ra_time_start( time );
   if ( !defined $self->ra_position )
      {
      $self->ra_position( 0 );
      $self->ra_to_limit( );
      }
   #
   #
   foreach ( qw/focus_0 focus_1 dec_drive / )
      {
      my $position = sprintf '%s_position', $_;
      my $to_limit = sprintf '%s_to_limit', $_;
      my $last_direction = sprintf '%s_last_direction', $_;
      my $motor_on = sprintf 'clear_%s_motor_on', $_;

      $self->driver->motor_setup( uc $_, 'IN' );
      $self->$last_direction( 'IN' );

      $self->driver->motor_off( uc $_ );
      $self->$motor_on;

      if ( !defined $self->$position )
         {
         $self->$position( 0 );
         $self->$to_limit;
         }
      }
   #
   # If we're not using DEC clamp, turn on DEC drive for holding
   if ( !$self->dec_clamp_use )
      {
      $self->driver->motor_on( 'DEC_DRIVE' );
      $self->set_dec_drive_motor_on;
      }
   $self->shutter_0_close;
   $self->shutter_1_close;
   $self->_chiller_last_time( [gettimeofday] );
   $self->chiller_off;
   }

sub _load_positions
   {
   my $self = shift;


   if ( -e $self->position_file )
      {
      my %ini;
      tie %ini, 'Config::IniFiles', ( -file => $self->position_file );

      $self->ra_position( $ini{save}{ra_position} );
      $self->dec_clamp_position( $ini{save}{dec_clamp_position} );
      $self->dec_drive_position( $ini{save}{dec_drive_position} );
      $self->focus_0_position( $ini{save}{focus_0_position} );
      $self->focus_1_position( $ini{save}{focus_1_position} );
      #
      # We delete the position file after reading so that if the program
      # exits abnormally, we won't pick up an incorrect position next time
      # around
      delete $ini{save};
      $ini{save} = { };
      (tied %ini)->RewriteConfig;
      }
   }

sub _save_positions
   {
   my $self = shift;

   my %ini;

   if ( -e $self->position_file )
      {
      tie %ini, 'Config::IniFiles', ( -file => $self->position_file );
      }
   else
      {
      tie %ini, 'Config::IniFiles';
      (tied %ini)->SetFileName( $self->position_file );
      }

   $ini{save} = { };
   $ini{save}{ra_position} = $self->ra_current_second
      if defined $self->ra_current_second && $self->ra_current_second >= 0;
   $ini{save}{dec_clamp_position} = $self->dec_clamp_position
      if defined $self->dec_clamp_position && $self->dec_clamp_position >= 0;
   $ini{save}{dec_drive_position} = $self->dec_drive_position
      if defined $self->dec_drive_position && $self->dec_drive_position >= 0;
   $ini{save}{focus_0_position} = $self->focus_0_position
      if defined $self->focus_0_position && $self->focus_0_position >= 0;
   $ini{save}{focus_1_position} = $self->focus_1_position
      if defined $self->focus_1_position && $self->focus_1_position >= 0;

   (tied %ini)->RewriteConfig;
   untie %ini;
   }

sub _crater
   {
   my $self = shift;
   my $str = shift;

   logerr "Failed: $str";
   logerr "Shutting down the system";
   die "\nFailed: $str\nShutting down the system";
   }

#
# We do this as Time::Object doesn't seem to work (install) on newer
# systems.  DateTime provides similar functionality, but requires more
# setup
sub _get_gmtime
   {
   my $self = shift;

   my %gmtime = ( 'second' => 0, 'minute' => 1, 'hour' => 2,
                  'day' => 3, 'month' => 4, 'year' => 5 );

   my @time = gmtime;
   my %data = map{ $_ => $time[ $gmtime{ $_ } ] } keys %gmtime;
   $data{ 'year' } += 1900;
   $data{ 'month' }++;
   $data{ 'time_zone' } = 'GMT';

   my $gmtime = DateTime->new( %data );

   return $gmtime;
   }

=over

=item B<All Off>

=over

=item C<$tc-E<gt>all_off>

Turns off all motors (ra, both focus, dec drive and dec_clamp).  Generally
not required to be called by the user as when a TASS::Control object
is deleted, part of the process is to shut down the system automatically.

=item I<For Example>

=for example begin

 $tc->all_off;

=for example end

=back

=back

=cut

sub all_off
   {
   my $self = shift;
   my $status = 1;

   $status &= $self->driver->motor_ra( 'OFF' );
   $status &= $self->driver->motor_off( 'FOCUS_0' );
   $status &= $self->driver->motor_off( 'FOCUS_1' );
#   $status &= $self->driver->motor_off( 'DEC_DRIVE' );
#   $status &= $self->driver->motor_off( 'DEC_CLAMP' );
   $self->ra_speed( 0 );
   $self->clear_dec_drive_motor_on;
   $self->clear_focus_0_motor_on;
   $self->clear_focus_1_motor_on;

   $self->_crater( 'all_off' ) if !$status;
   }

sub DESTROY
   {
   my $self = shift;
   my $status = 1;

   if ( defined $self->driver )
      {
      $self->chiller_off;
      $status &= $self->driver->motor_ra( 'OFF' );
      $status &= $self->driver->motor_off( 'FOCUS_0' );
      $status &= $self->driver->motor_off( 'FOCUS_1' );
#      $status &= $self->driver->motor_off( 'DEC_DRIVE' );
#      $status &= $self->driver->motor_off( 'DEC_CLAMP' );
      #
      # Save the positions
      $self->_save_positions;

      logtrc INFO, "\t%5d bad transfers", $self->_bad_transfer;
      logtrc INFO, "\t%5d overexposures", $self->_overexposure;
      logtrc INFO, "\t%5d bias images", $self->_bias_count;
      logtrc INFO, "\t%5d dark images", $self->_dark_count;
      logtrc INFO, "\t%5d object images", $self->_object_count;
      if ( $self->chiller_use )
         {
         logtrc INFO, "\t%5d chiller on time",
                      $self->_chiller_time_on / 60;
         logtrc INFO, "\t%5d chiller off time",
                      $self->_chiller_time_off / 60;
         if ( $self->_chiller_time_on + $self->_chiller_time_off )
	    {
            logtrc INFO, "\t%5d%% chiller duty cycle",
                         100 * $self->_chiller_time_on /
                         ( $self->_chiller_time_on
                         + $self->_chiller_time_off);
	    }
         }
      }

   if ( !$self->_child )
      {
      local *SOCKET = $self->_socket;
      print SOCKET "QUIT\n";
      wait;
      }

   $self->clear_driver;
   }

=over

=item B<Dec Clamp Close>

=over

=item C<$tc-E<gt>dec_clamp_close>

Closes the DEC clamp by driving the motor in the amount specified
in the configuration variable dec_clamp_in.  Only closes the clamp if
the current position is known to be open.  Turns off the DEC drive motor
after closure is complete.  Leaves the DEC clamp motor off.

=item I<For Example>

=for example begin

 $tc->dec_clamp_close;

=for example end

=back

=back

=cut

sub dec_clamp_close
   {
   my $self = shift;
   my $status = 1;
   #
   # If we're out, do the commands, otherwise, NOP it
   if ( $self->dec_clamp_position > 0 )
      {
      $status &= $self->driver->motor_on( 'DEC_CLAMP' );
      $status &= $self->driver->motor_setup( 'DEC_CLAMP', 'IN' );
      $self->_crater( 'dec_clamp_close setup' ) if !$status;

      $status &= $self->driver->motor_move( 'DEC_CLAMP', $self->dec_clamp_in );
      $status &= $self->driver->motor_off( 'DEC_CLAMP' );
      $self->_crater( 'dec_clamp_close move' ) if !$status;

      $self->dec_clamp_position( 0 );
      #
      # If the dec drive motor is not off, turn it off now that the clamp is on
      $status &= $self->driver->motor_off( 'DEC_DRIVE' )
         if $self->dec_drive_motor_on;
      $self->_crater( 'dec_clamp_close dec motor off' ) if !$status;

      $self->clear_dec_drive_motor_on;
      }
   }

=over

=item B<Dec Clamp Open>

=over

=item C<$tc-E<gt>dec_clamp_open>

Opens the DEC clamp by driving the motor out the amount specified
in the configuration variable dec_clamp_out.  Only opens the clamp if
the current position is known to be closed.  Turns on the DEC drive motor
before the open is executed.  Leaves the DEC clamp motor off.

=item I<For Example>

=for example begin

 $tc->get_clamp_open;

=for example end

=back

=back

=cut

sub dec_clamp_open
   {
   my $self = shift;
   my $status = 1;
   #
   # If we're in, do the commands, otherwise NOP it
   if ( $self->dec_clamp_position == 0 )
      {
      #
      # If the dec drive motor is not on, turn it on now that the clamp
      # will be off
      $status &= $self->driver->motor_on( 'DEC_DRIVE' )
         if !$self->dec_drive_motor_on;
      $self->_crater( 'dec_clamp_open dec motor on' ) if !$status;

      $self->set_dec_drive_motor_on;

      $status &= $self->driver->motor_on( 'DEC_CLAMP' );
      $status &= $self->driver->motor_setup( 'DEC_CLAMP', 'OUT' );
      $self->_crater( 'dec_clamp_open setup' ) if !$status;

      $status &= $self->driver->motor_move( 'DEC_CLAMP', $self->dec_clamp_out );
      $status &= $self->driver->motor_off( 'DEC_CLAMP' );
      $self->_crater( 'dec_clamp_open move' ) if !$status;

      $self->dec_clamp_position( $self->dec_clamp_out );
      }
   }

=over

=item B<Dec Drive Move Steps>

=over

=item C<$tc-E<gt>dec_drive_move_steps( SCALAR )>

Moves DEC drive motor the specified number of steps.  Positive steps move
South declination, negative steps North.  Will open and close dec clamp,
but only if the clamp is closed.  If the clamp is open, it will remain open.
Program will exit if commanded to move below 0 or above the configurable
max range parameter dec_drive_range.

=item I<For Example>

=for example begin

 $tc->dec_drive_move_steps( 500 );

=for example end

=back

=back

=cut

sub dec_drive_move_steps
   {
   my $self = shift;
   my( $steps ) = @_;
   my $status = 1;
   my $direction = $steps > 0 ? 'IN' : 'OUT';
   my $clamped = ($self->dec_clamp_position == 0);

   $steps = int( $steps );

   return if $steps == 0;
#   $steps += $self->dec_drive_backlash
#      if $direction ne $self->dec_drive_last_direction;

   $self->_crater( 'dec_drive_move_steps: asking to move too far' )
      if !$self->_driving_to_limit
         && (  ($self->dec_drive_position + $steps) > $self->dec_drive_range
            || ($self->dec_drive_position + $steps) < 0);
   #
   # Open the clamp
   $self->dec_clamp_open if $clamped && $self->dec_clamp_use;
   #
   # Make sure the motor is on
   $status &= $self->driver->motor_on( 'DEC_DRIVE' )
      if !$self->dec_drive_motor_on;
   $self->_crater( 'dec_drive_move_steps dec motor on' ) if !$status;
   $self->set_dec_drive_motor_on;
   #
   # Move it
   $status &= $self->driver->motor_setup( 'DEC_DRIVE', $direction );
#      if $self->dec_drive_last_direction ne $direction;

   $status &= $self->driver->motor_move( 'DEC_DRIVE', abs $steps );
   $self->_crater( 'dec_drive_move_steps' ) if !$status;

   $self->dec_drive_last_direction( $direction );
   $self->dec_drive_position( $self->dec_drive_position + $steps );
   #
   # Close the clamp
   $self->dec_clamp_close if $clamped && $self->dec_clamp_used;
   #
   # make sure the motor if off
#   $status &= $self->driver->motor_off( 'DEC_DRIVE' )
#      if $self->dec_drive_motor_on;
#   $self->_crater( 'dec_drive_move_steps dec motor off' ) if !$status;
#   $self->clear_dec_drive_motor_on;
   }

=over

=item B<Dec Drive Move to Elevation>

=over

=item C<$tc-E<gt>dec_drive_move_to_elev( SCALAR )>

Moves DEC drive to the specified absolute elevation (given in degrees).
Level South is 0 degrees, with 90 straight up and 180 North.  See
L<dec_drive_move_steps> for specifics.  Uses dec_drive_steps_degree for
conversion to steps.

=item I<For Example>

=for example begin

 $tc->dec_drive_move_to_elev( 70 );

=for example end

=back

=back

=cut

sub dec_drive_move_to_elev
   {
   my $self = shift;
   my( $elev ) = @_;
   #
   # First off, figure out where we are
   my $cur_elev =
      ($self->dec_drive_position - $self->dec_drive_level_from_limit)
      / $self->dec_drive_steps_degree;
   my $steps = ($elev - $cur_elev) * $self->dec_drive_steps_degree;
   $self->dec_drive_move_steps( $steps );
   }

sub dec_drive_move_elev_rel
   {
   my $self = shift;
   my $rel_elev = shift;

   my $steps = $rel_elev * $self->dec_drive_steps_degree;
   $self->dec_drive_move_steps( $steps );
   }

=over

=item B<Dec Drive Move Degree>

=over

=item C<$tc-E<gt>dec_drive_move_degree( SCALAR )>

Moves DEC drive relative to the current position the number of degrees given.
See L<dec_drive_move_steps> for specifics.  Uses dec_drive_steps_degree
for conversion to steps.  Negative amounts move South, positive North.

=item I<For Example>

=for example begin

 $tc->dec_drive_move_degree( -1 );

=for example end

=back

=back

=cut

sub dec_drive_move_degree
   {
   my $self = shift;
   my( $degree ) = @_;

   my $steps = $degree * $self->dec_drive_steps_degree;
   $self->dec_drive_move_steps( $steps );
   }

=over

=item B<Dec Drive Move to Degree>

=over

=item C<$tc-E<gt>dec_drive_move_to_degree( SCALAR )>

Moves DEC drive to the specified absolute declination degree.  See
L<dec_drive_move_steps> for specifics.  Uses dec_drive_steps_degree for
conversion to steps.

=item I<For Example>

 $tc->dec_drive_move_to_degree( 0 );

=for example begin

=for example end

=back

=back

=cut

#
# RSC fixme
#
sub dec_drive_move_to_degree
   {
   my $self = shift;
   my( $dec ) = @_;

   my $steps =   ($dec - $self->dec_drive_current_degree)
               * $self->dec_drive_steps_degree;
   $self->dec_drive_move_steps( $steps );
   }

=over

=item B<Dec Drive Current Degree>

=over

=item C<SCALAR = $tc-E<gt>dec_drive_current_degree>

Returns the current declination degree.  Uses latitude,
dec_drive_level_from_limit and dec_drive_steps_degree along with the
current position.

=item I<For Example>

=for example begin

 $location = $tc->dec_drive_current_degree;

=for example end

=back

=back

=cut

#
# RSC fixme
#
sub dec_drive_current_degree
   {
   my $self = shift;

   my $dec = $self->latitude - 90
             + ($self->dec_drive_position - $self->dec_drive_level_from_limit)
             / $self->dec_drive_steps_degree;
   return $dec;
   }

=over

=item B<Dec Drive to Limit>

=over

=item C<$tc-E<gt>dec_drive_to_limit>

Moves DEC drive to the South declination limit switch.  If the limit is
currently tripped, moves North declination for a maximum of 500 steps until
it is off the limit.  If the 500 step limit is reached, the program exists.
Next, move South in steps of 100 until either the limit switch is reached or
dec_drive_range is exceeded.  Again, if it moves to far, the program exists.
Once the limit is reached, steps of 1 are used moving North until the limit
is cleared, and then move South 1 step at a time until the limit is
reached one last time.  The last move will give one measure of the backlash.

=item I<For Example>

=for example begin

 $tc->dec_drive_to_limit;

=for example end

=back

=back

=cut

sub dec_drive_to_limit
   {
   my $self = shift;

   my $clamped = ($self->dec_clamp_position == 0);

   #
   # Open the clamp
   $self->dec_clamp_open if $clamped && $self->dec_clamp_use;

   $self->driver->motor_setup( 'DEC_DRIVE', 'IN' );
   $self->driver->motor_on( 'DEC_DRIVE' )
      if !$self->dec_drive_motor_on;
   $self->set_dec_drive_motor_on;

   my $steps = $self->_motor_to_limit( "dec_drive", 25, 100 );

   #
   # Close the clamp
   $self->dec_clamp_close if $clamped && $self->dec_clamp_use;

   return $steps;
   }

sub _motor_to_limit
   {
   my $self = shift;
   my $motor = shift;
   my $bulk_steps = shift;
   my $max_steps = shift;

   my $status = 1;
   my $backlash = 0;
   my $steps = 0;
   my $range = $motor . "_range";
   my $position = $motor . "_position";
   my $at_limit = $motor . "_at_limit";
   my $move_steps = $motor . "_move_steps";

   $self->set__driving_to_limit;
   #
   # And move out of the limit if necessary
   while (  ($self->$at_limit)
         && ($steps < $max_steps)
         )
      {
      $self->$move_steps( $bulk_steps );
      $steps += $bulk_steps;
      }
   if ( $steps >= $max_steps )
      {
      $self->$position( undef );
      $self->_crater( $motor
         . "_to_limit: could not get off limit in $steps steps ($bulk_steps)")
      }
   #
   # And move on in
   $steps = 0;
   while (  (!$self->$at_limit)
         && ($steps < $self->$range)
         )
      {
      $self->$move_steps( -$bulk_steps );
      $steps += $bulk_steps;
      }
   $self->_crater( $motor
      . "_to_limit: could not find limit in $steps steps ($bulk_steps)" )
      if $steps >= $self->$range;
   #
   # And back out again, 1 step at a time
   $steps = 0;
   while (  ($self->$at_limit)
         && ($steps < $self->$range)
         )
      {
      $self->$move_steps( 1 );
      ++$steps;
      }
   $self->_crater( $motor
      . "_to_limit: could not get off limit in $steps steps (1)")
      if $steps >= $self->$range;
   #
   # And back in again, 1 step at a time
   $steps = 0;
   while (  (!$self->$at_limit)
         && ($steps < $self->$range)
         )
      {
      $self->$move_steps( -1 );
      ++$steps;
      ++$backlash;
      }
   --$backlash;
   $self->_crater( $motor
      . "_to_limit: could not find limit in $steps steps (1)" )
      if $steps >= $self->$range;

   $steps = $self->$position;
   $self->$position( 0 );

   logtrc INFO, "Missed tracking $motor by $steps steps";
   logtrc INFO, "\tbacklash might be $backlash";

   $self->clear__driving_to_limit;

   return $steps;
   }

=over

=item B<Dec Drive at Limit>

=over

=item C<SCALAR = $tc-E<gt>dec_drive_at_limit>

Returns if the DEC drive is at the limit switch or not.  Returns 1 and 0
respectively.

=item I<For Example>

=for example begin

 $limit = $tc->dec_drive_at_limit;

=for example end

=back

=back

=cut

sub dec_drive_at_limit
   {
   my $self = shift;
   my( $status, %limits ) = $self->driver->read_limits;
   $self->_crater( 'dec_drive_at_limit' ) if !$status;
   return $limits{ 'DEC_DRIVE' };
   }

=over

=item B<Delay>

=over

=item C<$tc-E<gt>delay( SCALAR )>

Method to use if any delays are required.  Floating point seconds accepted.
Will check for RA moving beyond range in either direction during delay.  If
excessive RA movement is detected, the program will exit.

=item I<For Example>

=for example begin

 $tc->delay( 4.5 );

=for example end

=back

=back

=cut

sub delay
   {
   my $self = shift;
   my $time = shift;

   my $start = time;
   my $left;
   do {
# The check was causing problems for some reason.  So, don't use it.
      $self->ra_at_limit;
      $self->chiller_control;
      $left = $time - (time - $start);
      if ( $left > 9 )
         {
         #
         # We will check for ra going to far just before each 9 second sleep,
         # as it's possible the user/program may do something intelligent
         # when the delay is complete.
         $self->ra_check_position( 'delay' );
         sleep 9;
         }
      elsif ( $left > 0 )
         {
         sleep $left;
         }
      } while ( $left > 9 );
   }

=over

=item B<RA Motor ON>

=over

=item C<$tc-E<gt>ra_on>

Turns on the RA motor to the specified speed.  If no speed is given, turns
on to sidereal rate.  Negative speed will move RA opposite sidereal.  Speed
has to be give in powers of 2.  Does no checking for current location of
RA, so traveling where you may not want to go is possible.

=item C<$tc-E<gt>ra_on( SCALAR )>

=item I<For Example>

=for example begin

 $tc->ra_on;
 $tc->ra_on( -32 );

=for example end

=back

=back

=cut

sub ra_on
   {
   my $self = shift;
   my $speed = shift || 1;
   my $status = 1;

   $status &= $self->driver->motor_ra( $speed );
   my $time = time;
   $self->ra_position( $self->ra_position
                     + ($time - $self->ra_time_start) * $self->ra_speed );
   $self->ra_time_start( $time );
   $self->ra_speed( $speed );
   $self->_crater( "ra_on: $speed" )
      if !$status;
   }

=over

=item B<RA Motor Off>

=over

=item C<$tc-E<gt>ra_off>

Turns off RA motor.

=item I<For Example>

=for example begin

 $tc->ra_off;

=for example end

=back

=back

=cut

sub ra_off
   {
   my $self = shift;
   my $status = 1;

   $status &= $self->driver->motor_ra( 'OFF' );
   my $time = time;
   $self->ra_position( $self->ra_position
                     + ($time - $self->ra_time_start) * $self->ra_speed );
   $self->ra_speed( 0 );
   $self->_crater( "ra_off" )
      if !$status;
   }

=over

=item B<RA to Limit>

=over

=item C<$tc-E<gt>ra_to_limit>

=item C<$tc-E<gt>ra_to_limit( SCALAR )>

Moves RA drive into the limit flag.  If a single argument is specified,
that is the starting speed and direction.  The direction given may be positive
if the RA drive is beyond the flag sensor on the 'wrong' side.  The RA limit
flag is checked as fast as possible during this operation.  Once the flag is
seen, the speed is halved and reversed, then when the flag is cleared,
the speed is halved and reversed again.  This occurs until the speed is -1 and
the flag is encountered for the final time.  By finding the limit switch in
this manner, a high speed move is used while still maintaining accuracy in
finding the limit switch position.  If the flag is not found within the
configured range of travel, the module will exit with an error logged.

=item I<For Example>

=for example begin

 $tc->ra_to_limit;
 $tc->ra_to_limit( -16 );
 $tc->ra_to_limit( 8 );

=for example end

=back

=back

=cut

sub ra_to_limit
   {
   my $self = shift;
   my $status = 1;
   my $speed = shift || ( -1 * $self->ra_max_speed );

   my $sidereal = 0;
   my $time;
   #
   #
   if ( $self->ra_current_second < 0 )
      {
      $self->ra_on( abs( $speed ) );
      while ( ($self->ra_current_second < 0) && !$self->ra_at_limit ) { };
      }
   $self->ra_on( abs( $speed ) ) if $self->ra_at_limit;
   $time = [gettimeofday];
   while( $self->ra_at_limit )
      {
      $self->_crater(
         "ra_to_limit: could not get off flag in 500 sidereal seconds" )
         if abs( tv_interval( $time ) * $speed ) > 500;
      }

   my $last_speed = 2;
   while ( abs $speed > $last_speed )
      {
      $self->ra_on( $speed );
      $time = [gettimeofday];
      while( !$self->ra_at_limit )
         {
         $self->_crater(
            sprintf 'ra_to_limit: did not find flag in %d sidereal seconds',
                    abs( tv_interval( $time ) * $speed ) )
            if abs( tv_interval( $time ) * $speed ) > $self->ra_range;
         }

      $speed /= -2;

      $self->ra_on( $speed );
      $time = [gettimeofday];
      while( $self->ra_at_limit )
         {
         $self->_crater(
       sprintf 'ra_to_limit: could not get off flag in %d sidereal seconds',
               $self->ra_current_second )
            if abs( tv_interval( $time ) * $speed ) > $self->ra_range;
         }

      $speed /= -2;
      }

   $self->ra_on( -1 );
   $time = [gettimeofday];
   while( !$self->ra_at_limit )
         {
         $self->_crater(
          sprintf 'ra_to_limit: could not find flag in %d sidereal seconds',
                  $self->ra_current_second )
            if tv_interval( $time ) > $self->ra_range;
         }

   $self->ra_off;

   logtrc INFO, "Missed tracking ra by %.2f seconds", $self->ra_position;
   my $off = $self->ra_position;

   $self->ra_position( 0 );

   return $off;
   }

=over

=item B<RA Time Left>

=over

=item C<SCALAR = $tc-E<gt>ra_time_left>

Returns the number of sidereal seconds availabel until the configured range
of travel is exceeded.

=item I<For Example>

=for example begin

 $seconds = $tc->ra_time_left;

=for example end

=back

=back

=cut

sub ra_time_left
   {
   my $self = shift;

   return   $self->ra_range - $self->ra_position
          - $self->ra_speed * (time - $self->ra_time_start);
   }

=over

=item B<RA Current Degree>

=over

=item C<SCALAR = $tc-E<gt>ra_current_degree>

Returns the current RA position is decimal degrees.

=item I<For Example>

=for example begin

 $where = $tc->ra_current_degree;

=for example end

=back

=back

=cut

sub ra_current_degree
   {
   my $self = shift;

   my $mjd = now2mjd;
   my $ra = $mjd - ($self->ra_level_from_limit - $self->ra_time_left) / 86400;
   return turn2deg( mjd2lst( $ra, $self->_t_longitude ) ),
   }

=over

=item B<RA Current Hour>

=over

=item C<$tc-E<gt>ra_current_hour>

Returns the current RA position is decimal hour angle.

=item I<For Example>

=for example begin

 $hour = $tc->ra_current_hour;

=for example end

=back

=back

=cut

sub ra_current_hour
   {
   my $self = shift;

   my $mjd = now2mjd;
   my $ra = $mjd - (  $self->ra_level_from_limit - $self->ra_time_left )
                   / 86400;
   return turn2str( mjd2lst( $ra, $self->_t_longitude ), 'H', 2 ),
   }

=over

=item B<RA Current Second>

=over

=item C<$tc-E<gt>ra_current_second>

Returns the current second position from the limit switch.

=item I<For Example>

=for example begin

 $second = $tc->ra_current_second;

=for example end

=back

=back

=cut

sub ra_current_second
   {
   my $self = shift;

   return $self->ra_position + $self->ra_speed * (time - $self->ra_time_start);
   }

=over

=item B<RA Check Position>

=over

=item C<SCALAR = $tc-E<gt>ra_check_position>

Checks if the current RA position is beyond the configured range at either
end of travel.  If RA is beyond either limit, it is moved to the limit and then
turned RA off.  Returns 1 if RA is not beyond limits, 0 if it is (was).
Primarily used as an internal sanity check, but exported for the insane.

=item I<For Example>

=for example begin

 $status = $tc->ra_check_position;

=for example end

=back

=back

=cut

sub ra_check_position
   {
   my $self = shift;
   my( $name ) = @_;
   my $status = 1;

   if (  ($self->ra_speed != 0)
      && (  ($self->ra_current_second < 0)
         || ($self->ra_current_second > $self->ra_range)
         )
      )
      {
      logerr '%s: Went past %d limit at %d sidereal seconds, shutting down RA',
             $self->ra_current_second < 0 ? 0 : $self->ra_range,
             $name, $self->ra_current_second;
      if ( $self->ra_current_second < 0 )
         {
         $self->ra_on( $self->ra_max_speed );
         while ( $self->ra_current_second < 0 ) { };
         }
      elsif ( $self->ra_current_second > $self->ra_range )
         {
         $self->ra_on( -1 * $self->ra_max_speed );
         while ( $self->ra_current_second < 0 ) { };
         }
      $self->ra_off;
      $status = 0;
      }
   return $status;
   }

=over

=item B<RA to Meridian>

=over

=item C<$tc-E<gt>ra_to_meridian>

=item C<$tc-E<gt>ra_to_meridian( SCALAR, SCALAR )>

Moves RA to the meridian.  Two optional parameters may be given which specify
the offset from meridian (in seconds) and the speed to move at.

=item I<For Example>

=for example begin

 $tc->ra_to_meridian;
 $tc->ra_to_meridian( 30, 8 );
 $tc->ra_to_meridian( undef, 8 );

=for example end

=back

=back

=cut

sub ra_to_meridian
   {
   my $self = shift;
   my $offset = shift || 0;
   my $speed = shift || $self->ra_max_speed;
   
   $self->ra_move_second(   $self->ra_level_from_limit
                          - $self->ra_current_second
                          + $offset, $speed );
   }

=over

=item B<RA Move Second>

=over

=item C<$tc-E<gt>ra_move_second( SCALAR )>

=item C<$tc-E<gt>ra_move_second( SCALAR, SCALAR )>

Moves RA relative the number of seconds given, with the speed to move
an optional parameter.

=item I<For Example>

=for example begin

 $tc->ra_move_second( 459 );
 $tc->ra_move_second( 459, -8 );
 $tc->ra_move_second( -459, 8 );

=for example end

=back

=back

=cut

sub ra_move_second
   {
   my $self = shift;
   my $final  = shift;
   my $speed = shift || $self->ra_max_speed;

   my $prev_speed = $self->ra_speed;
   my $delay = $final / $speed;
   $self->ra_on( ($delay < 0 ? -1 : 1) * $speed );
   $self->delay( abs $delay );
   if ( $prev_speed != 0 )
      {
      $self->ra_on( $prev_speed );
      }
   else
      {
      $self->ra_off;
      }
   }

=over

=item B<RA at Limit>

=over

=item C<SCALAR = $tc-E<gt>ra_at_limit>

Returns if the RA drive is at the limit switch or not.  Returns 1 and 0
respectively.

=item I<For Example>

=for example begin

 $limit = $tc->ra_at_limit;

=for example end

=back

=back

=cut

sub ra_at_limit
   {
   my $self = shift;
   my( $status, %limits ) = $self->driver->read_limits;
   $self->_crater( 'ra_at_limit'  ) if !$status;
   return $limits{ 'RA' };
   }

=over

=item B<Current RA Range>

=over

=item C<(SCALAR, SCALAR) = $tc-E<gt>current_ra_range>

Returns the current RA "reach" in degrees.

=item I<For Example>

=for example begin

 ($start, $end ) = $tc->current_ra_range;

=for example end

=back

=back

=cut

sub current_ra_range
   {
   my $self = shift;

   my $mjd = now2mjd;
   my $r1 = $mjd - $self->ra_level_from_limit / 86400;
   my $r2 = $mjd + $self->ra_level_from_limit / 86400;

   logtrc INFO, sprintf "RA range %s to %s",
                turn2deg( mjd2lst( $r1, $self->_t_longitude ) ),
                turn2deg( mjd2lst( $r2, $self->_t_longitude ) );

   return turn2deg( mjd2lst( $r1, $self->_t_longitude ) ),
          turn2deg( mjd2lst( $r2, $self->_t_longitude ) );
   }

=over

=item I<RA Move to Degree>

=over

=item C<SCALAR = $tc->E<gt>ra_move_to_degree( SCALAR, SCALAR )>

Moves RA to the specified absolute location.  The second parameter is optinal
and specifies the speed to move at.  Tries to be smart by shorting (or
lengthening) the move time to hit the moving target.  Checks that the
requested location can be reached, and returns 1 if it can, 0 if not.
The requested speed should be a power of 2.

=item I<For Example>

=for example begin

 $status = $tc->ra_move_to_degree( 52, 64 );

=for example end

=back

=back

=cut

sub ra_move_to_degree
   {
   my $self = shift;
   my $ra = shift;
   my $speed = shift || $self->ra_max_speed;
   my $status = 1;

   my( $ra_seconds, $ra_time );

   #
   # Now find the ra distance
   $ra_seconds = ($self->ra_current_degree - $ra) * 3600 / 15 ;
   $ra_time = abs( $ra_seconds );
   #
   # If we're wrapping...
   if ( $ra_time > (2 * $self->ra_level_from_limit) )
      {
      $ra_seconds += 360 * 3600 / 15 if ( $ra_seconds < 0 );
      $ra_seconds -= 360 * 3600 / 15 if ( $ra_seconds > 0 );
      }
   logdbg INFO, sprintf "RA Move %f at %d",
                $ra_seconds, $speed;
   $self->ra_move_second( $ra_seconds, $speed );
   }

=over

=item B<Move to RA/DEC location>

=over

=item C<SCALAR = $tc-E<gt>move_to_ra_dec( SCALAR, SCALAR )>

This routine will point the telescope to the location given (in decimal
degrees), attempting to move both RA and DEC at the same time if possible,
to reduce transit times.
Returns '1' if the location can be reached or '0' if not (typically RA not
within range).

=item I<For Example>

=for example begin

 $status = $tc->move_to_ra_dec( 10, 22 );

=for example end

=back

=back

=cut

sub move_to_ra_dec
   {
   my $self = shift;
   my( $ra, $dec ) = @_;

   my $prev_speed = $self->ra_speed;

   #
   # Check that we can reach where we're wanted to go
   my @ra_range = $self->current_ra_range;
   if ( $ra_range[0] > $ra_range[1] )
      {
      return 0 if ($ra > $ra_range[0]) and ($ra < $ra_range[1]);
      }
   else
      {
      return 0 if ($ra < $ra_range[0]) or ($ra > $ra_range[1]);
      }
   
   my $dec_steps;
   my $ra_seconds;
   my $dec_time = 0;
   my $ra_time = 0;
   $dec_steps =   ($dec - $self->dec_drive_current_degree)
                * $self->dec_drive_steps_degree;
   $dec_time += abs( $dec_steps ) * 0.025; #$self->driver->_motors( 'DEC_DRIVE' )
   #
   # Calculate how long it will take to open the clamp, move and
   # then close the clamp provided the clamp is closed.
   if (  ($self->dec_clamp_position == 0)
      && ($dec_time != 0)
      && ($self->dec_clamp_use)
      )
      {
      $dec_time
         += $self->dec_clamp_out * 0.025; #($self->driver->_motors( 'DEC_CLAMP' ))[4];
      $dec_time
         += $self->dec_clamp_in * 0.025; #($self->driver->_motors( 'DEC_CLAMP' ))[4];
      }
   #
   # Now find the ra distance
   $ra_seconds = ($ra - $self->ra_current_degree) * 3600 / 15 ;
   $ra_time = abs( $ra_seconds );
   #
   # If we're wrapping...
   if ( $ra_time > (2 * $self->ra_level_from_limit) )
      {
      $ra_seconds += 360 * 3600 / 15 if ( $ra_seconds < 0 );
      $ra_seconds -= 360 * 3600 / 15 if ( $ra_seconds > 0 );
      $ra_time = abs( $ra_seconds );
      }

   #
   # And calculate the log to get the RA speed we will be using
   $dec_time = .0001 if !$dec_time;
   my $log = log( $ra_time / $dec_time ) / log( 2 );
   my $ra_speed = int( 2 ** int( $log ) );
   $ra_speed = $self->ra_max_speed
      if $ra_speed > $self->ra_max_speed;
   $ra_speed *= $ra_seconds < 0 ? -1 : 1;
   logdbg INFO, sprintf "Move to RA/DEC %f %f %f",
                $ra_time / $dec_time, $log, $ra_speed;
   #
   # Turn on RA, move dec and then goto the correct RA
   $self->ra_on( $ra_speed );
   $self->dec_drive_move_to_degree( $dec );
   $self->ra_on( $prev_speed );
   $self->ra_move_to_degree( $ra );
   return 1;
   }

=over

=item B<Focus Move Steps>

=over

=item C<$tc-E<gt>focus_0_move_steps( SCALAR )>

=item C<$tc-E<gt>focus_1_move_steps( SCALAR )>

Moves the specifiec camera focus the specified number of steps

=item I<For Example>

=for example begin

 $tc->focus_0_move_steps( 25 );
 $tc->focus_1_move_steps( 50 );

=for example end

=back

=back

=cut

sub focus_0_move_steps
   {
   my $self = shift;
   my $steps = shift;

   my $motor_on = $self->focus_0_motor_on;
   my $status = 1;
   my $motor = 'FOCUS_0';

   if ( !$motor_on )
      {
      $status &= $self->driver->motor_on( $motor );
      $self->set_focus_0_motor_on;
      $self->_crater( 'focus_0_move_steps: could not turn on motor' )
         if !$status;
      }
   $self->_motor_move_steps( $motor, $steps );
   if ( !$motor_on )
      {
      $status &= $self->driver->motor_off( $motor );
      $self->clear_focus_0_motor_on;
      $self->_crater( 'focus_0_move_steps: could not turn off motor' )
         if !$status;
      }
   return;
   }

=over

=item B<Focus Move To Position>

=over

=item C<$tc-E<gt>focus_0_move_to_position( SCALAR )>

=item C<$tc-E<gt>focus_1_move_to_position( SCALAR )>

Moves the specifiec camera focus to the indicated position

=item I<For Example>

=for example begin

 $tc->focus_0_move_to_position( 500 );
 $tc->focus_1_move_to_position( 350 );

=for example end

=back

=back

=cut

sub focus_0_move_to_position
   {
   my $self = shift;
   my $position = shift;

   my $steps = $position - $self->focus_0_position;

   $self->focus_0_move_steps( $steps ) if $steps != 0;

   return;
   }

=over

=item B<Focus to Limit>

=over

=item C<$tc-E<gt>focus_0_to_limit>

=item C<$tc-E<gt>focus_1_to_limit>

Moves the specified camera focus to the limit switch

=item I<For Example>

=for example begin

 $tc->focus_0_to_limit;
 $tc->focus_1_to_limit;

=for example end

=back

=back

=cut

sub focus_0_to_limit
   {
   my $self = shift;

   my $status = 1;

   $status &= $self->driver->motor_on( 'FOCUS_0' );
   $self->set_focus_0_motor_on;
   $self->_crater( 'focus_0_to_limit: could not turn on motor' )
      if !$status;
   $self->_focus_to_limit( 0 );
   $status &= $self->driver->motor_off( 'FOCUS_0' );
   $self->clear_focus_0_motor_on;
   $self->_crater( 'focus_0_to_limit: could not turn on motor' )
      if !$status;
   }

=over

=item B<Focus at Limit>

=over

=item C<SCALAR = $tc-E<gt>focus_0_at_limit>

=item C<SCALAR = $tc-E<gt>focus_1_at_limit>

Returns if the specified camera focus is at the limit switch or not.
Returns 1 and 0 respectively.

=item I<For Example>

=for example begin

 $limit = $tc->focus_0_at_limit;
 $limit = $tc->focus_1_at_limit;

=for example end

=back

=back

=cut

sub focus_0_at_limit
   {
   my $self = shift;
   my( $status, %limits ) = $self->driver->read_limits;
   $self->_crater( 'focus_0_at_limit' ) if !$status;
   return $limits{ 'CAMERA_0' };
   }

sub _focus_to_limit
   {
   my $self = shift;
   my $camera = shift;

   my $status = 1;

   my $motor = "FOCUS_$camera";

   $self->_motor_to_limit( "focus_$camera", 5, 100 );
   return;
   }

sub _motor_move_steps
   {
   my $self = shift;
   my $motor = shift;
   my $steps = shift;

   my $status = 1;
   my $direction = $steps < 0 ? 'IN' : 'OUT';
   my $position = sprintf '%s_position', lc $motor;
   my $range = sprintf '%s_range', lc $motor;
   my $last_direction = sprintf '%s_last_direction', lc $motor;

   $self->_crater(
      sprintf '%s_move_steps: asking to move too far', lc $motor )
      if !$self->_driving_to_limit
         && (  ($self->$position + $steps) > $self->$range
            || ($self->$position + $steps) < 0);

   $status &= $self->driver->motor_setup( $motor, $direction );
#      if $self->$last_direction ne $direction;
   $self->$last_direction( $direction );
   $status &= $self->driver->motor_move( $motor, abs( $steps ) );
   $self->_crater(
      sprintf '%s_move_steps failed the move ', lc $motor )
      if !$status;

   $self->$position( $self->$position + $steps );
   return;
   }

sub focus_1_at_limit
   {
   my $self = shift;
   my( $status, %limits ) = $self->driver->read_limits;
   $self->_crater( 'focus_1_at_limit' ) if !$status;
   return $limits{ 'CAMERA_1' };
   }

sub focus_1_to_limit
   {
   my $self = shift;

   my $status = 1;

   $status &= $self->driver->motor_on( 'FOCUS_1' );
   $self->set_focus_1_motor_on;
   $self->_crater( 'focus_1_to_limit: could not turn on motor' )
      if !$status;
   $self->_focus_to_limit( 1 );
   $status &= $self->driver->motor_off( 'FOCUS_1' );
   $self->clear_focus_1_motor_on;
   $self->_crater( 'focus_1_to_limit: could not turn on motor' )
      if !$status;
   }

sub focus_1_move_steps
   {
   my $self = shift;
   my $steps = shift;

   my $motor_on = $self->focus_1_motor_on;
   my $status = 1;
   my $motor = 'FOCUS_1';

   if ( !$motor_on )
      {
      $status &= $self->driver->motor_on( $motor );
      $self->set_focus_1_motor_on;
      $self->_crater( 'focus_1_move_steps: could not turn on motor' )
         if !$status;
      }
   $self->_motor_move_steps( $motor, $steps );
   if ( !$motor_on )
      {
      $status &= $self->driver->motor_off( $motor );
      $self->clear_focus_1_motor_on;
      $self->_crater( 'focus_1_move_steps: could not turn off motor' )
         if !$status;
      }
   return;
   }

sub focus_1_move_to_position
   {
   my $self = shift;
   my $position = shift;

   my $steps = $position - $self->focus_1_position;

   $self->focus_1_move_steps( $steps ) if $steps != 0;

   return;
   }

=over

=item B<Shutter 0 Open>

=over

=item C<$tc-E<gt>shutter_0_open>

Opens the camera 0 sutter to the value specified in the config file
for "shutter_0_open".

=item I<For Example>

=for example begin

 $tc->shutter_0_open;

=for example end

=back

=back

=cut

sub shutter_0_open
   {
   my $self = shift;

   $self->_crater( 'shutter_0_open' )
      if !$self->driver->maservo( 'SERVO_0', $self->shutter_0_open_pos );
   }

=over

=item B<Shutter 1 Open>

=over

=item C<$tc-E<gt>shutter_1_open>

Opens the camera 1 sutter to the value specified in the config file
for "shutter_1_open".

=item I<For Example>

=for example begin

 $tc->shutter_1_open;

=for example end

=back

=back

=cut

sub shutter_1_open
   {
   my $self = shift;

   $self->_crater( 'shutter_1_open' )
      if !$self->driver->maservo( 'SERVO_1', $self->shutter_1_open_pos );
   }

=over

=item B<Shutter 0 Close>

=over

=item C<$tc-E<gt>shutter_0_close>

Closes the camera 0 sutter to the value specified in the config file
for "shutter_0_close".

=item I<For Example>

=for example begin

 $tc->shutter_0_close;

=for example end

=back

=back

=cut

sub shutter_0_close
   {
   my $self = shift;

   $self->_crater( 'shutter_0_close' )
      if !$self->driver->maservo( 'SERVO_0', $self->shutter_0_close_pos );
   }

=over

=item B<Shutter 1 Close>

=over

=item C<$tc-E<gt>shutter_1_close>

Closes the camera 1 sutter to the value specified in the config file
for "shutter_1_close".

=item I<For Example>

=for example begin

 $tc->shutter_1_close;

=for example end

=back

=back

=cut

sub shutter_1_close
   {
   my $self = shift;

   $self->_crater( 'shutter_1_close' )
      if !$self->driver->maservo( 'SERVO_1', $self->shutter_1_close_pos );
   }

=over

=item B<Get File Names>

=over

=item C<(SCALAR, SCALAR) = $tc-E<gt>get_file_names>

Returns the file names for the last set of images taken.

=item I<For Example>

=for example begin

 ($camera_0_file, $camera_1_file) = $tc->get_file_names;

=for example end

=back

=back

=cut

sub get_file_names
   {
   my $self = shift;
   #
   # To generate the file name, we generate the integer portion...
   my $file_base = int(  ( $self->_picture_time - 50000 )
                       * 10000 + 0.5 );
   #
   # ...then we append the file suffix...
   $file_base .= $self->file_suffix;
   #
   # ...then we prepend the file site, camera,
   # processing status and reserved char
   my( $file_1, $file_2 ) = ( $file_base, $file_base );
   $file_1 = $self->site_code . 'vr_' . $file_base;
   $file_2 = $self->site_code . 'ir_' . $file_base;
   return $file_1, $file_2;
   }

sub _inb
   {
   my $self = shift;

   my $command = 'inb ' . shift;
   my $result = `$command`;
   chomp $result;
   return $result;
   }

sub _outb
   {
   my $self = shift;

#   my $port = shift;
#   my $command;
   my $command = 'outb ' . join ' ', @_;
   logdbg INFO, 'Calling outb: ' . join ' ', @_;
#   map { 'outb $port $_' } @_;
#   for @_ { 'outb $port $_' };
#   return `$command`;
   }

sub _wait_for_transfer
   {
   my $self = shift;
   my $start = [gettimeofday];

   sleep 9;

   $self->ra_at_limit;
   my $last_send = [gettimeofday];
   while( !(oct( $self->_inb( '0x300' ) ) & 0x80))
      {
      if ( tv_interval( $last_send ) > 9 )
         {
         $self->ra_check_position( '_wait_for_transfer' );
         $self->ra_at_limit;
         $last_send = [gettimeofday];
         }
      sleep( 1 );
      last if tv_interval( $start ) > 50;
      }
   if ( !(oct( $self->_inb( '0x300' ) ) & 0x80)
      && (tv_interval( $start ) > 50)
      )
      {
      $self->_bad_transfer_incr;
      logerr "Memory card transfer didn't complete in %.2f seconds (%d)",
             tv_interval( $start ), $self->_bad_transfer;
      $self->driver->load_register( 'MPX', 0 );
#
# Real recursive loop here.  Should it be fixed and use the clear?
#      $self->clear_ccds;
      return 0;
      }
   return 1;
   }

sub _gen_pic_info
   {
   my $self = shift;

   my( $status, $value );

   $self->pic_info_num( 'ra', $self->ra_current_degree );
   $self->pic_info_num( 'ra_left', $self->ra_time_left );
   $self->pic_info_num( 'ra_limit', $self->ra_current_second );
   $self->pic_info_num( 'CRVAL2', $self->ra_current_degree );
   $self->pic_info_num( 'dec', $self->dec_drive_current_degree );
   $self->pic_info_num( 'CRVAL1', $self->dec_drive_current_degree );
   for my $adc ( 0..31 )
      {
      ( $status, $value ) = $self->driver->read_adc( $adc );
      $self->_crater( "gen_pic_info adc $adc" ) if !$status;
      $self->pic_info_num( sprintf( 'adc_%02d', $adc ), $value );
      }
   }

=over

=item B<CCD Read Reset>

=item C<$tc-E<gt>ccd_read_reset>

=over

=item I<For Example>

=for example begin

 $tc->ccd_read_reset;

=for example end

=back

=back

=cut

sub ccd_read_reset
   {
   my $self = shift;

   logtrc INFO, 'Executing read reset';
#   $self->_outb( '0x300', '0x80', '0x81', '0x90', '0x00' );
   $self->_outb( '0x300', '0x80', '0x91' );
   }

sub ccd_write_reset
   {
   my $self = shift;

   logtrc INFO, 'Executing write reset';
   $self->_outb( '0x300', '0x00' );
   $self->driver->pulse( 17 );

#   $self->_outb( '0x300', '0x80', '0x91', '0x00' );
   }

sub ccd_take_picture
   {
   my $self = shift;

   $self->ccd_write_reset;
   logtrc INFO, 'Transfering to memory card';
   $self->driver->pulse( 4 );
   my $status = $self->_wait_for_transfer;
   $self->_outb( '0x300', '0x80' );
   $self->_last_picture_end( [gettimeofday] );

   return $status;
   }

=over

=item B<Clear CCDs>

=over

=item C<$tc-E<gt>clear_ccds>

Executes a clearing scan for the CCDs by resetting the memory card and
initiating a image transfer.  Waits for the transfer to complete before
returning.  Will retry the operation one time by resetting the memory
card and initiating the image transfer again after 60 seconds of not
seeing the image transfer complete.  If the retry transfer fails, the
program will be exited.

=item I<For Example>

=for example begin

 $tc->clear_ccds;

=for example end

=back

=back

=cut

sub clear_ccds
   {
   my $self = shift;

   logtrc INFO, 'Executing clearing scan';
   if ( !$self->ccd_take_picture )
      {
      $self->_crater( 'clear_ccds: failed second clear' )
         unless !$self->ccd_take_picture;
      }
   }

sub _download_server
   {
   my $self = shift;
   my $command;

   local *SOCKET = $self->_socket;
   my( $id, $file_0, $file_1 );
   my $quit = 0;

   logdbg INFO, 'Download server ready';

   while ( $command = <SOCKET> )
      {
      chomp $command;
      logdbg DEBUG, "Download server: $command";
      SWITCH: for ( $command )
         {
         /^START\s+(.*)$/ && do
            {
            $self->_download_pictures( $1, $file_0, $file_1 );
            print SOCKET "DONE\n";
            last SWITCH;
            };
         /^FILES\s+(\S+)\s+(\S+)$/ && do
            {
            $file_0 = $1;
            $file_1 = $2;
            last SWITCH;
            };
         /^KEY_FLT\s+(\S+)\s+(\S+)$/ && do
            {
            $self->pic_info_num( $1, $2 );
            last SWITCH;
            };
         /^KEY_STR\s+(\S+)\s+(.*)$/ && do
            {
            $self->pic_info_str( $1, $2 );
            last SWITCH;
            };
         /^QUIT$/ && do
            {
            $quit = 1;
            logdbg DEBUG, "Received quit";
            last SWITCH;
            };
         logtrc WARN, "Received unknown command: $command";
         }
      last if $quit;
      }
   logdbg INFO, "Exiting download server";
   }

sub _download_pictures
   {
   my $self = shift;
   my $id = shift;
   my @names = @_;

   logtrc INFO, "$id: Starting download";
   my $time = [gettimeofday];
   my $comment;
   my $status = 0;
   logdbg INFO, "$id: Downloading %s and %s", @names;
   my $cmd = sprintf "downfits -n %d -m %d -c %s -d %s",
                      $self->naxis1, $self->naxis2, @names;

   system $cmd;
   use Astro::FITS::CFITSIO qw( :longnames :constants );

   my $fptr0 = Astro::FITS::CFITSIO::open_file(
      $names[0], Astro::FITS::CFITSIO::READWRITE( ), $status );
   my $fptr1 = Astro::FITS::CFITSIO::open_file(
      $names[1], Astro::FITS::CFITSIO::READWRITE( ), $status );

   for ( sort $self->pic_info_num_keys )
      {
      $comment = undef;
      $comment = $self->pic_info_comments( $_ )
         if $self->pic_info_comments_exists( $_ );
      $fptr0->write_key_flt( $_,
                             $self->pic_info_num( $_ ), -8,
                             defined $comment ? $comment : undef, $status );
      $fptr1->write_key_flt( $_,
                             $self->pic_info_num( $_ ), -8,
                             defined $comment ? $comment : undef, $status );
      }

   for ( sort $self->pic_info_str_keys )
      {
      $comment = undef;
      $comment = $self->pic_info_comments( $_ )
         if $self->pic_info_comments_exists( $_ );
      $fptr0->write_key_str( $_,
                             $self->pic_info_str( $_ ),
                             defined $comment ? $comment : undef, $status );
      $fptr1->write_key_str( $_,
                             $self->pic_info_str( $_ ),
                             defined $comment ? $comment : undef, $status );
      }
   $fptr0->write_key_str( 'FILTER',
                          $self->camera_0_filter,
                          undef, $status );
   $fptr1->write_key_str( 'FILTER',
                          $self->camera_1_filter,
                          undef, $status );
   $fptr0->close_file( $status );
   $fptr1->close_file( $status );
   logtrc INFO, "$id: Completed download in %.2f seconds",
      tv_interval( $time );
   }

sub _download
   {
   my $self = shift;
   my $id = shift;

   local *SOCKET = $self->_socket;

   printf SOCKET "FILES %s %s\n", $self->get_file_names;
   for ( $self->pic_info_num_keys )
      {
      printf SOCKET "KEY_FLT %s %f\n", $_, $self->pic_info_num( $_ );
      }
   for ( $self->pic_info_str_keys )
      {
      printf SOCKET "KEY_STR %s %s\n", $_, $self->pic_info_str( $_ );
      }
   print SOCKET "START $id\n";
   $self->set__in_progress;
   }

sub _wait_for_download
   {
   my $self = shift;

   return if !$self->_in_progress;

   my $done = 0;
   my $socket = IO::Select->new( $self->_socket );
   local *SOCKET = $self->_socket;

   logtrc INFO, "Waiting for previous download to finish";

   do
      {
      my @handles = $socket->can_read( 8 );
      if ( scalar( @handles ) == 0 )
         {
         $self->ra_check_position( '_wait_for_download' );
         $self->ra_at_limit;
         }
      else
         {
         my $cmd = <SOCKET>;
         chomp $cmd;
         $done = 1 if $cmd =~ /^DONE$/;
         }
      }
   while ( !$done );
   $self->clear__in_progress;
   }

=over

=item B<Take Object Pictures>

=over

=item C<$tc-E<gt>take_object_pictures( SCALAR, SCALAR, SCALAR, SCALAR )>

Takes object pictures.  The first two arguments are mandatory and specify
the number of pictures to take and their desired exposure length.  The
second two arguments are optional (either or both can be 'undef'), and
specify the RA and DEC distance to move B<before> the image is taken.
RA is used in the call L<ra_to_meridian>, and can be either positive or
negative.  This is most usefull to effect a track and rewind for the given
picture set.
DEC is used in the call L<dec_drive_move_degree>, which will move the
given number of degrees for each picture.  Using the movement parameters
will add to the dark time of the object exposure, with the maximum
being dictated only by the time it takes to execute the given moves.
The last argument is the 'ra_jitter' flag, and indicates if the ra
drive should be turned off during the picture transfer (48.9 seconds) or not.

=item I<For Example>

=for example begin

 $tc->take_object_pictures( 10, 60 );
 $tc->take_object_pictures( 10, 60, undef, 4 );
 $tc->take_object_pictures( 10, 60, -30, undef );
 $tc->take_object_pictures( 10, 100, -50, -1 );

=for example end

=back

=back

=cut

sub take_object_pictures
   {
   my $self = shift;
   my( $count, $duration, $ra, $dec, $ra_jitter ) = @_;

   my $time;

   #
   # If the last picture ended real recently, we don't
   # have to execute a clearing scan
   if ( tv_interval( $self->_last_picture_end ) < $self->max_object_dark_time )
      {
      logtrc INFO, "No clearing scan needed for object: %.2f",
                   tv_interval( $self->_last_picture_end );
      }
   else
      {
      $self->_wait_for_download;
      $self->clear_ccds;
      }

   for ( 1..$count )
      {
      $self->ra_on;
      if ( defined $dec )
         {
         logtrc INFO, "O $_: Moving dec: relative $dec degrees";
         $self->dec_drive_move_degree( $dec );
         }

      if ( defined $ra )
         {
         logtrc INFO, "O $_: Moving ra: meridian with offset $ra";
         $self->ra_to_meridian( $ra, $self->ra_max_speed );
         }

      $self->pic_info_num( 'darktime',
                           tv_interval( $self->_last_picture_end ) );
      my $time = [gettimeofday];
      $self->shutter_0_open;
      $self->shutter_1_open;
      my $shutter_time = tv_interval( $time );

      $self->_picture_time( now2mjd );

      $time = [gettimeofday];

      $self->_gen_pic_info;
      $self->pic_info_str( 'imagetyp', 'OBJECT' );

      $self->_wait_for_download;

      if( ($duration - tv_interval( $time ) - $shutter_time) > 0 )
         {
         logtrc INFO, "O $_: Waiting %.2f seconds for exposure",
                      $duration - tv_interval( $time ) - $shutter_time;
         $self->delay( $duration - tv_interval( $time ) - $shutter_time );
         }
      else
         {
         logerr "O $_: Last download took too long";
         logerr "O $_: Overexposure on images %s and %s by %d seconds",
                $self->get_file_names,
                -$duration + tv_interval( $time ) + $shutter_time;
         $self->_overexposure_incr;
         }

      $self->shutter_0_close;
      $self->shutter_1_close;

      $self->pic_info_num( 'exptime', tv_interval( $time ) );
      $self->pic_info_num( 'epoch', (gmtime)[5] + 1900 );

      my $gmtime = $self->_get_gmtime;
      if ( defined $ra_jitter && $ra_jitter )
         {
         $self->ra_off;
         }

      $time = [gettimeofday];
      next if !$self->ccd_take_picture;

      my $exp = DateTime::Duration->new(
                   'seconds' => $self->pic_info_num( 'exptime' ) / 2 );
      $gmtime = $gmtime - $exp;
      $self->pic_info_str( 'date-obs', $gmtime->datetime );
      $self->pic_info_str( 'utstart', $gmtime->hms );

      logtrc INFO, "O $_: Exposure length was %.2f seconds",
                   $self->pic_info_num( 'exptime' );

      logtrc INFO, "O $_: Transfer took %.2f seconds", tv_interval( $time );
      $self->_object_count_incr;

      $self->_download( "O $_" );
      }
   }

=over

=item B<Take Dark Pictures>

=over

=item C<$tc-E<gt>take_dark_pictures( SCALAR, SCALAR )>

Takes dark images.  The first argument specifies the image count and the
second the desired exposure length.

=item I<For Example>

=for example begin

 $tc->take_dark_pictures( 2, 60 );

=for example end

=back

=back

=cut

sub take_dark_pictures
   {
   my $self = shift;
   my( $count, $duration ) = @_;

   my $time;

   #
   # If the last picture ended in within a good time,
   # use the time completed so far on the exposure for a dark exposure
   # Just make sure we have at some time to do what we need to
   if (  ( tv_interval( $self->_last_picture_end ) )
       < ($duration - $self->dark_diff)
      )
      {
      logtrc INFO, "No clearing scan needed for dark: %.2f",
                   tv_interval( $self->_last_picture_end );
      }
   else
      {
      $self->_wait_for_download;
      $self->clear_ccds;
      }

   $self->pic_info_num( 'darktime', 0 );

   for ( 1..$count )
      {
      $self->_picture_time( now2mjd );

      $self->_gen_pic_info;
      $self->pic_info_str( 'imagetyp', 'DARK' );

      $self->_wait_for_download;

      if( ($duration - tv_interval( $self->_last_picture_end ) ) > 0 )
         {
         logtrc INFO, "D $_: Waiting %.2f seconds for exposure",
                      $duration - tv_interval( $self->_last_picture_end );
         $self->delay( $duration - tv_interval( $self->_last_picture_end ) );
         }
      else
         {
         logerr "D $_: Last download took too long";
         logerr "D $_: Overexposure on images %s and %s by %d seconds",
                $self->get_file_names,
                -$duration + tv_interval( $self->_last_picture_end );
         $self->_overexposure_incr;
         }

      $self->pic_info_num( 'exptime',
                           tv_interval( $self->_last_picture_end ) );
      my $gmtime = $self->_get_gmtime;
      $time = [gettimeofday];
      next if !$self->ccd_take_picture;

      my $exp = DateTime::Duration->new(
                   'seconds' => $self->pic_info_num( 'exptime' ) / 2 );
      $gmtime = $gmtime - $exp;
      $self->pic_info_str( 'date-obs', $gmtime->datetime );
      $self->pic_info_str( 'utstart', $gmtime->hms );

      logtrc INFO, "D $_: Exposure length was %.2f seconds",
                   $self->pic_info_num( 'exptime' );

      logtrc INFO, "D $_: Transfer took %.2f seconds", tv_interval( $time );
      $self->_dark_count_incr;

      $self->_download( "D $_" );
      }
   }

=over

=item B<Take Bias Pictures>

=over

=item C<$tc-E<gt>take_bias_pictures( SCALAR )>

Takes bias images.  The argument specifies the number of bias images to
take.

=item I<For Example>

=for example begin

 $tc->take_bias_pictures( 1 );

=for example end

=back

=back

=cut

sub take_bias_pictures
   {
   my $self = shift;
   my( $count ) = @_;

   my $time;

   $self->pic_info_num( 'darktime', 0 );

   for ( 1..$count )
      {
      $self->_wait_for_download;
      $self->clear_ccds;

      $self->_picture_time( now2mjd );

      my $gmtime = $self->_get_gmtime;
      $time = [gettimeofday];
      next if !$self->ccd_take_picture;

      $self->pic_info_num( 'exptime', 0 );

      my $exp = DateTime::Duration->new(
                   'seconds' => $self->pic_info_num( 'exptime' ) / 2 );
      $gmtime = $gmtime - $exp;
      $self->pic_info_str( 'date-obs', $gmtime->datetime );
      $self->pic_info_str( 'utstart', $gmtime->hms );

      logtrc INFO, "B $_: Exposure length was %.2f seconds",
                   $self->pic_info_num( 'exptime' );

      $self->_gen_pic_info;
      $self->pic_info_str( 'imagetyp', 'BIAS' );

      logtrc INFO, "B $_: Transfer took %.2f seconds", tv_interval( $time );
      $self->_bias_count_incr;

      $self->_download( "B $_" );
      $self->chiller_control;
      }
   }

=over

=item B<DAC Start>

=over

=item C<$tc-E<gt>dac_start>

Loads the 'dac_??_start' values from the configuration file into the DACs.

=item I<For Example>

=for example begin

 $tc->dac_start;

=for example end

=back

=back

=cut

sub dac_start
   {
   my $self = shift;

   map { my $x = sprintf "dac_%02d_start", $_;
         $self->driver->write_dac( $_, $self->$x ) } (0..15);
   }

=over

=item B<Dac End>

=over

=item C<$tc-E<gt>dac_end>

Loads the 'dac_??_end' values from the configuration file into the DACs.

=item I<For Example>

=for example begin

 $tc->dac_end;

=for example end

=back

=back

=cut

sub dac_end
   {
   my $self = shift;

   map { my $x = sprintf "dac_%02d_end", $_;
         $self->driver->write_dac( $_, $self->$x ) } (0..15);
   }

=over

=item B<Power Control>

=over

=item C<$tc-E<gt>power_control( SCALAR )>

Loads the POWER register with the given value.  Can be used for chiller on
(0x01) and chiller off (0x01)

=item I<For Example>

=for example begin

 $tc->power_control( 0x01 );

=for example end

=back

=back

=cut

sub chiller_on
   {
   my $self = shift;
   my $value = shift;

   $self->set__chiller_on;
   $self->driver->load_register( 'POWER', 0x01 );
   logtrc INFO, "Chiller on";
   }

sub chiller_off
   {
   my $self = shift;
   my $value = shift;

   $self->clear__chiller_on;
   $self->driver->load_register( 'POWER', 0x00 );
   logtrc INFO, "Chiller off";
   }

sub chiller_control
   {
   my $self = shift;  

   return if !$self->chiller_use;

   my $water_temp_adc = 0;
   my $air_temp_adc = 4;

   my( $water_status, $water_temp )
      = $self->driver->read_adc( $water_temp_adc );
   my( $air_status, $air_temp )
      = $self->driver->read_adc( $air_temp_adc );
   
      #
      # Advance the timers
      if ( $self->_chiller_on )
         {
         $self->_chiller_time_on(
              $self->_chiller_time_on
            + tv_interval( $self->_chiller_last_time ) );
         $self->_chiller_current_run_time(
              $self->_chiller_current_run_time
            + tv_interval( $self->_chiller_last_time ) );
         }
      else
         {
         $self->_chiller_time_off(
              $self->_chiller_time_off
            + tv_interval( $self->_chiller_last_time ) );
         $self->_chiller_current_run_time(
              $self->_chiller_current_run_time
            + tv_interval( $self->_chiller_last_time ) );
         }

   if ( $air_status && $water_status )
      {
      #
      # Calculate the correct off temperature
      my $off_delta = $air_temp - $self->delta_temp;
      my $off_temp = $off_delta < $self->min_temp ?
                        $self->min_temp : $off_delta;
      #
      # Calculate the correct on temperature
      my $on_temp = $off_temp + $self->dead_band;

      my $switch_chiller = 0;
      my $perc = 100 * $self->_chiller_time_on /
         (  $self->_chiller_time_on + $self->_chiller_time_off );
      logtrc INFO, "Chiller %d %.1f - %.1f - %.1f %.1f%%",
                   $self->_chiller_on,
                   $on_temp, $water_temp, $off_temp, $perc;
      #
      # If the chiller is on, check that it should be
      if ( $self->_chiller_on )
         {
         if (  ($self->_chiller_current_run_time > $self->min_time)
            || ($self->_chiller_time_on > $self->max_time_on)
            )
            {
            if (  (  ($self->_chiller_current_run_time > $self->max_time_on)
                  || ($perc > $self->duty_cycle) )
               )
	       {
               $switch_chiller = 1;
	       }
            elsif (  ($self->_chiller_current_run_time > $self->min_time)
                  && ($water_temp < $off_temp)
                  )
	       {
               $switch_chiller = 1;
	       }
            }
         }
      else
         {
         if (  ($water_temp > $on_temp)
            && ($perc < $self->duty_cycle)
            && ($self->_chiller_current_run_time > $self->min_time)
            )
	    {
            $switch_chiller = 1;
	    }
         }

      if ( $switch_chiller )
         {
         $self->_chiller_current_run_time( 0 );
         if ( $self->_chiller_on )
            {
            $self->chiller_off;
            }
         elsif ( !$self->_chiller_on )
            {
            $self->chiller_on;
            }
         }
      }
   $self->_chiller_last_time( [gettimeofday] );
   }

1;
__END__
