# $Name:  $
#
# $Log: Driver.pm,v $
# Revision 0.4  2005/01/16 03:25:59  robert
# Change SHUTTER to SERVO
# Add ability to ignore maservo errors
# Change some logging from INFO to DEBUG
#
# Revision 0.3  2003/01/25 19:17:40  robert
# Tweak some logging messages
#
# Revision 0.2  2002/12/23 04:07:18  robert
# Fix tests from previous ADC dependency read removals
# Fix _flush when simulating
# Add 1 second extra (2.5 total) wait during servo (shutter) moves
# Document load_register method
# Pretty up logging information
#
# Revision 0.1.1.1  2002/11/03 02:53:17  robert
#
#
package TASS::Control::Driver;

use 5.006;
use strict;
use warnings;

use Device::SerialPort;
use Time::HiRes qw( time usleep );
use Log::Agent;
use Log::Agent::Priorities qw( :LEVELS );
use Callback;
use Carp;
use POSIX qw( floor );

my $ID = q$Id: Driver.pm,v 0.4 2005/01/16 03:25:59 robert Exp $;
my $version = join ( ' ', ( split ( ' ', $ID ) )[ 1 .. 3 ] );
$version =~ s/,v\b//;
$version =~ s/(\S+)$/($1)/;
our $VERSION = $version;

=begin testing

use Log::Agent;
require Log::Agent::Driver::File;
require Log::Agent::Tag::Callback;

BEGIN{ use_ok( 'TASS::Control::Driver' ); }

my %attrib = 
   (
   'port'  => '/dev/ttyS1',
   'baud'  => '2400',
   '_command' => 'zFEDCBAw',
   'simulate' => 'false'
   );

my %attrib_new = 
   (
   'port'   => '/dev/ttyS2',
   'baud'   => '9600',
   '_command' => 'xABCDEFy',
   'simulate' => 'true'
   );

#
# Make sure we can create an instance with no hash involved
my $no_hash = TASS::Control::Driver->new;
ok( defined $no_hash, 'defined from new with no hash' );
undef $no_hash;

#
# Set up the logging
# unlink 'td.err', 'td.dbg', 'td.out';
logconfig( -driver => Log::Agent::Driver::File->make(
                         -prefix => 'TD',
                         -showpid => 1,
                         -channels => { 'error' => 'td.err',
                                        'output' => 'td.out',
                                        'debug' => 'td.dbg' } ),
           -debug => DEBUG
         );

my $hash = TASS::Control::Driver->new( %attrib );
ok( defined $hash, 'defined from new with hash' );

$hash->start;

#
# Test the read, write and clear methods
for my $key ( keys %attrib )
   {
   my $clear_key = "clear_$key";

   # Read
   is( $hash->$key, $attrib{ $key }, 'read of value worked' );
   # Clear and test the clear
   $hash->$clear_key;
   is( $hash->$key, undef, 'clearing of value worked' );
   # Write
   is( $attrib_new{ $key }, $hash->$key( $attrib_new{ $key } ),
       'writing value worked' );
   # Read after Write
   is( $hash->$key, $attrib_new{ $key }, 'read after write' );
   }

#
# Check to make sure the command was set correctly, and then, when preped,
# the commands morphs correctly (bytes 0 and 7 become a number
ok( $hash->_command =~ /^xABCDEFy$/, 'command set' );
$hash->_prep_command;
ok( $hash->_command =~ /^[0-9]{1}ABCDEF[0-9]{1}$/, '_prep_command' );
#
# Now, do the prep and check the command to make sure we're changing it every
# time.  It's possible this test will fail, but not likely
my $check = $hash->_command;
$hash->_prep_command;
ok( $hash->_command ne $check, '_prep_command different' )
   or diag <<__END_DUP__;
The test that just failed is working with 2 random digits (0-9).
It's possible that we actually just duplicated those two digits
across two runs, so please try <make test> again before calling
it a day, and a failure.  Thanks.

__END_DUP__
#
# Now, we'll set the response to the command for an 'easy' ok
$hash->_response( $hash->_command );
is( $hash->_check_response, 1, '_check_response' );
#
# Now, re-set the response back to the un-modified command for a failure
$hash->_response( $attrib{ '_command' } );
isnt( $hash->_check_response, 1, 'failing _check_response' );

#
# A simulated execute of the previous command set.  Makes sure the
# sim setup works
ok( $hash->_execute_command, '_execute_command' );

#
# Check the adc boundries for appropriate failures;
isnt( ($hash->read_adc( -1 ))[0], '-1 adc read' );
isnt( ($hash->read_adc( 32 ))[0], '32 adc read' );

#
# Go check some adc readings.
my $test = 'sim: adc simple read, pos 1';
logdbg DEBUG, "--- $test ---";
my( $status, $adc ) = $hash->read_adc( 2, 0x1000 );
ok( $status, "execute $test" );
is( sprintf( "%.5f", $adc ), '1310699.99939', "value $test" );

$test = 'sim: adc simple read, neg 1';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 31, 0xF000 );
ok( $status, "execute $test" );
is( sprintf( "%.5f", $adc ), '1310699.99939', "value $test" );

$test = 'sim: adc simple read, pos 10';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 17, 0x1000 );
ok( $status, "execute $test" );
is( sprintf( "%.4f", $adc ), '13106999.9939', "value $test" );

$test = 'sim: adc simple read, neg 10';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 24, 0xF000 );
ok( $status, "execute $test" );
is( sprintf( "%.4f", $adc ), '13106999.9939', "value $test" );

$test = 'sim: adc simple read, pos 100';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 18, 0x3000 );
ok( $status, "execute $test" );
is( sprintf( "%.3f", $adc ), '131069726.939', "value $test" );

$test = 'sim: adc simple read, neg 100';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 19, 0xF000 );
ok( $status, "execute $test" );
is( sprintf( "%.3f", $adc ), '131069726.939', "value $test" );

#
# Now, go check some more adc readings.  These depend on anothers
# reading, so we pass 2 values to use
$test = 'sim: adc complex read, pos 100, 100';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 0, 0x4000, 0x1000 );
ok( $status, "execute $test" );
is( sprintf( "%.3f", $adc ), '131069726.939', "value $test" );

$test = 'sim: adc complex read, neg 100, 100';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 1, 0xF000, 0xF000 );
ok( $status, "execute $test" );
is( sprintf( "%.3f", $adc ), '131069726.939', "value $test" );

$test = 'sim: adc complex read, pos 10, 100';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 25, 0x4000, 0x1000 );
ok( $status, "execute $test" );
is( sprintf( "%.4f", $adc ), '13106999.9939', "value $test" );

$test = 'sim: adc complex read, neg 10, 100';
logdbg DEBUG, "--- $test ---";
( $status, $adc ) = $hash->read_adc( 26, 0xE000, 0xF000 );
ok( $status, "execute $test" );
is( sprintf( "%.4f", $adc ), '13106999.9939', "value $test" );

#
# Here, we work on the limit switches
my %limits;
($status, %limits) = $hash->read_limits( 0x82 );
ok( $status, 'read limits' );
is( scalar keys %limits, 4, 'number of limits' );
is( $limits{ 'RA' }, 1, 'RA set' );
is( $limits{ 'DEC_DRIVE' }, 0, 'DEC_DRIVE not set' );
is( $limits{ 'CAMERA_1' }, 1, 'CAMERA_1 set' );
is( $limits{ 'CAMERA_0' }, 0, 'CAMERA_0 set' );

($status, %limits) = $hash->read_limits( 0x41 );
ok( $status, 'read limits' );
is( scalar keys %limits, 4, 'number of limits' );
is( $limits{ 'RA' }, 0, 'RA set' );
is( $limits{ 'DEC_DRIVE' }, 1, 'DEC_DRIVE not set' );
is( $limits{ 'CAMERA_1' }, 0, 'CAMERA_1 set' );
is( $limits{ 'CAMERA_0' }, 1, 'CAMERA_0 set' );

#
# Write some failing dacs here
isnt( $hash->write_dac( -1, 0 ), 'dac -1 write' );
isnt( $hash->write_dac( 16, 0 ), 'dac 16 write' );

#
# Lets try writing some good dacs now, and check the packet built
ok( $hash->write_dac( 0, -65000 ), 'dac 0 write' );
like( $hash->_command, qr/....(\?\?)./, 'write value FF' );
ok( $hash->write_dac( 15, 65000 ), 'dac 15 write' );
like( $hash->_command, qr/....(00)./, 'write value 00' );
ok( $hash->write_dac( 8, -11.0536 ), 'dac 8 write' );
like( $hash->_command, qr/....(\?>)./, 'write value FE' );
ok( $hash->write_dac( 9, 13.5886 ), 'dac 9 write' );
like( $hash->_command, qr/....(01)./, 'write value 01' );

#
# "Model Airplane Servo" testing.  Not much to do here, just make sure
# invalid channels fail and valid one don't
isnt( $hash->maservo( -1, 0 ), 'move servo -1' );
isnt( $hash->maservo( 24, 0 ), 'move servo 24' );
isnt( $hash->maservo( 'SERVO', 0 ), 'move servo "SERVO"' );
isnt( $hash->maservo( 2 ), 'move servo 2 with no position' );
isnt( $hash->maservo( 2, 256 ), 'move servo 2 with bad position 256' );
ok( $hash->maservo( 0, 255 ), 'move servo 0 position 255' );
ok( $hash->maservo( 'SERVO_0', 0 ), 'move servo "SERVO_0" position 0' );
ok( $hash->maservo( 'SERVO_1', 0 ), 'move servo "SERVO_0" position 0' );

#
# "Pulse" testing.  Presumably sends a pulse to whomever is listening
# on the channe.  Not much to do here, just check for invalid entries
# and valid entries
isnt( $hash->pulse( -1 ), 'pulse -1' );
isnt( $hash->pulse( 24 ), 'pulse -24' );
ok( $hash->pulse( 0 ), 'pulse 0' );
ok( $hash->pulse( 23 ), 'pulse 23' );

#
# Motor setup (direction) setting
isnt( $hash->motor_setup( 'FOCUS_I', 'why' ),
      'setup "FOCUS_I" for direction "why"' );
isnt( $hash->motor_setup( 'FOCUS_V', 2 ),
      'setup "FOCUS_V" for direction "2"' );
isnt( $hash->motor_setup( 'VW', 'OUT' ), 'setup "VW" for direction "OUT"' );
isnt( $hash->motor_setup( 'RA', 'OUT' ), 'setup "RA" for direction "OUT"' );
ok( $hash->motor_setup( 'FOCUS_1', 'IN' ),
    'setup "FOCUS_1" for direction "IN"' );
ok( $hash->motor_setup( 'FOCUS_0', 'OUT' ),
    'setup "FOCUS_0" for direction "OUT"' );
ok( $hash->motor_setup( 'DEC_DRIVE', '-1' ),
    'setup "DEC_DRIVE" direction "-1"' );
ok( $hash->motor_setup( 'DEC_CLAMP', '1' ),
    'setup "DEC_CLAMP" direction of "1"' );

#
# Motor move to limit testing
isnt( $hash->motor_to_limit( 'DEC_CLAMP' ), 'move to limit for "DEC_CLAMP"' );
ok( $hash->motor_to_limit( 'FOCUS_0' ), 'move to limit for "0"' );
ok( $hash->motor_to_limit( 'FOCUS_1' ), 'move to limit for "1"' );
ok( $hash->motor_to_limit( 'DEC_DRIVE' ), 'move to limit for "DRIVE"' );

#
# Moter on/off testing.  Again, just make sure what should work does,
# and what shouldn't doesn't
isnt( $hash->motor_on( 'BREAD' ), 'motor_on "BREAD"' );
isnt( $hash->motor_off( 'CAR' ), 'motor_off "CAR"' );
isnt( $hash->motor_on( 'RA' ), 'motor_on "RA"' );
isnt( $hash->motor_off( 'RA' ), 'motor_off "RA"' );

ok( $hash->motor_on( 'FOCUS_0' ), 'motor on "FOCUS_0"' );
ok( $hash->motor_off( 'FOCUS_0' ), 'motor on "FOCUS_0"' );
ok( $hash->motor_on( 'FOCUS_1' ), 'motor on "FOCUS_1"' );
ok( $hash->motor_off( 'FOCUS_1' ), 'motor on "FOCUS_1"' );
ok( $hash->motor_on( 'DEC_DRIVE' ), 'motor on "DEC_DRIVE"' );
ok( $hash->motor_off( 'DEC_DRIVE' ), 'motor on "DEC_DRIVE"' );
ok( $hash->motor_on( 'DEC_CLAMP' ), 'motor on "DEC_CLAMP"' );
ok( $hash->motor_off( 'DEC_CLAMP' ), 'motor on "DEC_CLAMP"' );

#
# Motor move testing
isnt( $hash->motor_move( 'blue', 1 ), 'moving motor "blue"' );
isnt( $hash->motor_move( 'DEC_CLAMP', -1 ),
      'moving motor "DEC_CLAMP" -1 steps' );
isnt( $hash->motor_move( 'DEC_DRIVE', MAX_STEPS ),
      'moving motor "DEC_CLAMP"'. MAX_STEPS . 'steps' );
ok( $hash->motor_move( 'FOCUS_0', 1 ), 'moving motor "FOCUS_0" 1 steps' );
ok( $hash->motor_move( 'FOCUS_1', 1 ), 'moving motor "FOCUS_1" 1 steps' );
ok( $hash->motor_move( 'DEC_CLAMP', 1 ), 'moving motor "DEC_CLAMP" 1 steps' );
ok( $hash->motor_move( 'DEC_DRIVE', 0xFFFF ),
    'moving motor "DEC_CLAMP"'. 0xFFFF . 'steps' );

#
# RA motor testing
isnt( $hash->motor_ra( 'bananna' ), 'motor ra "bananna"' );
ok( $hash->motor_ra( 'OFF', 'motor ra "OFF"' ) );
ok( $hash->motor_ra( '0', 'motor ra "0"' ) );
ok( $hash->motor_ra( 'ON', 'motor ra "ON"' ) );
ok( $hash->motor_ra( '1', 'motor ra "1"' ) );
ok( $hash->motor_ra( '2', 'motor ra "2"' ) );
ok( $hash->motor_ra( '4', 'motor ra "4"' ) );
ok( $hash->motor_ra( '8', 'motor ra "8"' ) );
ok( $hash->motor_ra( '16', 'motor ra "16"' ) );
ok( $hash->motor_ra( '32', 'motor ra "32"' ) );
ok( $hash->motor_ra( '64', 'motor ra "64"' ) );
ok( $hash->motor_ra( '128', 'motor ra "128"' ) );
ok( $hash->motor_ra( '-1', 'motor ra "-1"' ) );
ok( $hash->motor_ra( '-2', 'motor ra "-2"' ) );
ok( $hash->motor_ra( '-4', 'motor ra "-4"' ) );
ok( $hash->motor_ra( '-8', 'motor ra "-8"' ) );
ok( $hash->motor_ra( '-16', 'motor ra "-16"' ) );
ok( $hash->motor_ra( '-32', 'motor ra "-32"' ) );
ok( $hash->motor_ra( '-64', 'motor ra "-64"' ) );
ok( $hash->motor_ra( '-128', 'motor ra "-128"' ) );

#
# Load register stuff - should be a good catch all routine for everything
# I've forgotten
isnt( $hash->load_register( 'FOO', 0 ), 'load register "FOO" with 0' );
isnt( $hash->load_register( 'MPX', -1 ), 'load register "MPX" with -1' );
isnt( $hash->load_register( 'MOTOR', 256 ), 'load register "MOTOR" with 256' );
ok( $hash->load_register( 'MPX', 22 ), 'load register "MPX" with 22' );
ok( $hash->load_register( 'MOTOR', 56 ), 'load register "MPX" with 56' );

=end testing

=head1 NAME

TASS::Control::Driver - A perl modules which implements a low level serial
communication to control a TASS IV telescope.

=head1 SYNOPSIS

=for example begin

 use TASS::Control::Driver;

 $td = TASS::Control::Driver->new( %attributes );
 $status = $td->start;

 ($status, $value) = $td->read_adc( $adc );
 $status = $td->write_dac( $dac, $value );
 ($status, %limits) = $td->read_limits;

=for example end

=head1 DESCRIPTION

=head1 METHODS

=over

=item B<new>

=over

=item C<OBJREF = TASS::Control::Driver-E<gt>new>

=item C<OBJREF = TASS::Control::Driver-E<gt>new( HASH )>

Creates a Tass Driver object and returns a reference to said object.
Either takes a hash of L<attributes|"ATTRIBUTES"> or nothing as the agruments.

=item I<For Example>

=for example begin

 $td = TASS::Control::Driver->new( ( 'port' => '/dev/ttyS2',
                                     'baud' => '9600' ) );

=for example end

=item B<Simulation>

To place the object under simulation, set the simulation hash entry to 'true'.
Alternatively, the simulation attribute may be set as described in the
L<attributes|"ATTRIBUTES"> section sometime before calling the method I<start>.

=over

=item I<For Example>

=for example begin

 $td = TASS::Control::Driver->new( ( 'port' => '/dev/ttyS2',
                                     'baud' => '9600',
                                     'simulate' => 'true' ) );

=for example end

=back

=back

=cut

use Class::MethodMaker
   new_with_init  => 'new',
   new_hash_init  => '_hash_init',
   get_set        => [ qw/port baud simulate ignore_maservo_errors/ ],
   get_set        => [ qw/_last_send_time _command _response _device/ ],
   get_set        => [ qw/_extra_wait/ ],
   boolean        => [ qw/_started/ ],
   list           => [ qw/_check_chars/ ],
   counter        => [ qw/_command_count/ ],
   counter        => [ qw/_retry _noise _errors _timeouts _extra_chars/ ],
   get_set        => [ qw/max_retry/ ],
   hash           => [ qw/_adc_offset _adc_scale _adc_depend/ ],
   hash           => [ qw/_commands _pulses _maservo_names _registers/ ],
   hash           => [ qw/_limits _motors _ra_cmds _motor_directions/ ],
   get_set        => [ qw/_motors_on/ ];

#
# This is the delay which we must wait between sending any commands to the
# stamp - in seconds
use constant COMMAND_DELAY => 0.0;
#
# How long to wait for a response, in seconds
# not sure why 1.2 was here and .2 was commented out...
use constant RESPONSE_WAIT => 0.5;
#use constant RESPONSE_WAIT => 1.2;
use constant MAX_RETRY => 3;
#
# The leading response character
use constant RESPONSE_CHAR => 'S';
#
# How many DACs/ADCs there are
use constant MAX_DAC => 16;
use constant MAX_ADC => 32;
#
# How many "Model Airplane" servo's there are
use constant MAX_MASERVO_CHANNEL  => 24;
use constant MAX_MASERVO_POSITION => 256;
#
# How many Pulse channel's there are
use constant MAX_PULSE_CHANNEL => 24;
#
# The maximum number of steps allowed
use constant MAX_STEPS => 0x10000;
#
# The maximum number of steps the STAMP will move to a limit switch
use constant MAX_STEPS_TO_LIMIT => 5000;

sub init
   {
   my( $self, %args ) = @_;

   $self->clear_ignore_maservo_errors;

   $self->_hash_init( %args );

   logconfig( -tags => [ Log::Agent::Tag::Callback->make(
                            -callback => Callback->new( \&_build_tag ) ) ] );

   $self->_command_count_reset;
   $self->_extra_chars_reset;
   $self->_errors_reset;
   $self->_timeouts_reset;
   $self->_noise_reset;

   $self->_extra_wait( 0 );
   $self->max_retry( MAX_RETRY );
   #
   # Set up the commands and their OP codes
   $self->_commands(
      'LOADADAC',          '1',  # Only first byte is used for OP code
      'ADC',               '20',
      'PULSE',             '30',
      'SENSE',             '60',
      'MASERVO',           '70',
#      'STEPPER',           '80', currently unused
      'MOTOR_ON_OFF',      '81',
      'MOTOR_MOVE_STEPS',  '92',
      'MOTOR_MOVE_LIMIT',  '93',
      'MOTOR_SETUP',       '94'
      );
   #
   # Set up the known registers
   $self->_registers(
      'MPX',   '?2',
      'POWER', '51',
      'MOTOR', '59'
      );
   #
   # Set up motors and how to select them and mask them
   $self->_motors(
      'FOCUS_0',   [  0, 0x80, '7?', 0x01, 0.05 ],
      'FOCUS_1',   [  3, 0x10, '>?', 0x80, 0.05 ],
      'DEC_DRIVE', [  1, 0x40, ';?', 0x40, 0.025 ],
      'DEC_CLAMP', [  2, 0x20, '=?', 0x00, 0.025 ]
      );
   #
   # Just some stuff for motor directions...
   $self->_motor_directions(
      'IN', '0',
      '-1', '0',
      'OUT', '1',
      '1', '1',
      );
   #
   # Make sure the default motor states is all off, just for internal use...
   $self->_motors_on( 0x000 );
   #
   # RA commands are specific
   $self->_ra_cmds(
      'OFF',  '?8',
      '0',    '?8',
      '1',    '00',
      'ON',   '00',
      '2',    '20',
      '4',    '40',
      '8',    '60',
      '16',   '80',
      '32',   ':0',
      '64',   '<0',
      '128',  '>0',
      '-1',   '10',
      '-2',   '30',
      '-4',   '50',
      '-8',   '70',
      '-16',  '90',
      '-32',  ';0',
      '-64',  '=0',
      '-128', '?0'
      );
   #
   # Set up the limits, byte followed by bit (mask)
   $self->_limits(
      'CAMERA_1',  [ 5, 0x08 ],
      'DEC_DRIVE', [ 5, 0x04 ],
      'RA',        [ 6, 0x02 ],
      'CAMERA_0',  [ 6, 0x01 ],
      );
   #
   # Set up the pulse chanel commands
   $self->_pulses(
      0, '51',
      1, '71',
      2, '59',
      3, '79',
      4, '53',
      5, '73',
      6, '5;',
      7, '7;',
      8, '91',
      9, ';1',
      10, '99',
      11, ';9',
      12, '93',
      13, ';3',
      14, '9;',
      15, ';;',
      16, '=0',
      17, '?0',
      18, '=8',
      19, '?8',
      20, '=2',
      21, '?2',
      22, '=:',
      23, '?:'
      );
   #
   # Set up the maservo aliases
   $self->_maservo_names(
      'SERVO_0', 22,
      'SERVO_1', 14,
      'SERVO_2', 23,
      'SERVO_3', 13,
      'SERVO_4', 8,
      'SERVO_5', 12,
      'SERVO_6', 9,
      'SERVO_7', 11
      );
   #
   # Set up the default offset, then the specific offset values
   map { $self->_adc_offset( $_, 0 ) } 0..(MAX_ADC-1);
   map { $self->_adc_offset( $_, -2.73 ) } ( 0, 1, 3..5, 16, 18, 19 );

   #
   # Set up the default scale, then the specific scale values
   map { $self->_adc_scale( $_, 1 ) } 0..(MAX_ADC-1);
   map { $self->_adc_scale( $_, 10 ) } ( 9, 12, 17, 24..28 );
   map { $self->_adc_scale( $_, 100 ) } ( 0, 1, 3..5, 16, 18, 19 );

   #
   # And now, the dependency list
   # current form is:  x, [ y, o ]
   # where x is the adc to read, which depends on y, with an additive offset
   # of o
   $self->_adc_depend(
#      0, [ 3, 273 ],
#      1, [ 3, 273 ],
#      18, [ 16, 273 ],
#      19, [ 16, 273 ],
#      25, [ 24, 0 ],
#      26, [ 24, 0 ]
      );

   $self->_last_send_time( time );

   return $self;
   }

sub info
   {
   my $self = shift;

   logtrc INFO, "\t%5d commands", $self->_command_count;
   logtrc INFO, "\t%5d retries", $self->_retry;
   logtrc INFO, "\t%5d noise shots", $self->_noise;
   logtrc INFO, "\t%5d extra_characters", $self->_extra_chars;
   logtrc INFO, "\t%5d packet errors", $self->_errors;
   logtrc INFO, "\t%5d stamp timeouts", $self->_timeouts;
   }

sub DESTROY
   {
   my $self = shift;

   logtrc INFO, 'All done: ';
   $self->info;

   $self->_device->close if $self->_started;
   }

#
# Takes care of sending the command to the stamp, including not sending
# to fast to avoid overrunning the stamp
sub _send
   {
   my $self = shift;
   my $diff = time - $self->_last_send_time;

   #
   # Here we'll wait if necessary before sending the command
   sleep (COMMAND_DELAY - $diff) if $diff < COMMAND_DELAY;
   #
   # Send it here
   if ( !$self->simulate )
      {
      $self->_device->write( $self->_command );
      logdbg DEBUG, "<%s> Sent", $self->_command;
      }
   else
      {
      logdbg DEBUG, "<%s> Sim sent", $self->_command;
      }
   $self->_last_send_time( time );
   }

#
# Checks for characters outside the nominal to catch noise
sub _noise_check
   {
   my $self = shift;

   my $last_noise = -1;
   my $data = $self->_response;
   for ( 0..(length( $data ) - 1) )
      {
      $last_noise = $_ if ( (   ord( substr $data, $_, 1 ) < 0x30
                            || ord( substr $data, $_, 1 ) > 0x3F
                            )
                          && substr( $data, $_, 1 ) ne 'S'
                          );
      }
   if ( $last_noise != -1 )
      {
      $self->_noise_incr;
      logtrc WARN, "Flushed some noise - %d", $self->_noise;
      $self->_response( substr $data, $last_noise + 1 );
      logdbg DEBUG, "Noise from <%s> to <%s>", $data, $self->_response;
      }
   }

#
# Receive response data from the stamp
sub _receive
   {
   my $self = shift;
   my $char = shift;
   my $start = time;

   if ( ! $self->simulate )
      {
      #
      # If we are supposed to match a character...
      if ( defined $char )
         {
         #
         # Read out an initial char
         my( $count, $string ) = $self->_device->read( 1 );
         if ( $string ne $char )
            {
            do
               {
               logtrc WARN, "<%c> <0x%02X> received and tossed %d",
                  $string, $string, length $string
                  if $count == 1;
               ( $count, $string ) = $self->_device->read( 1 );
               }
#
# RSC fixme
#      while (  (time - $start < RESPONSE_WAIT)
            while (  ( (time - $self->_last_send_time)
                     < (RESPONSE_WAIT + $self->_extra_wait)
                     )
                  && ($string ne $char)
                  );
            }

         if ( $string ne $char )
            {
            logtrc WARN, "Timed out waiting for <$char>";
            }
         else
            {
            logdbg DEBUG, "Received <$char>";
            }
         }
      }

   #
   # Clear out the response so we can start appending to it, and then
   # wait 'till we read an entire response, or time out
   if ( ! $self->simulate )
      {
      $self->_response( '' );
      do
         {
         my( $count, $string ) = $self->_device->read( 8 );
         $self->_response( $self->_response . $string );
         $self->_noise_check;
         }
#
# RSC fixme
#      while (  (time - $start < RESPONSE_WAIT)
      while (  ( (time - $self->_last_send_time)
               < (RESPONSE_WAIT + $self->_extra_wait)
               )
            && (length $self->_response < 8)
            );
      }
   else
      {
      $self->_response( $self->_command );
      logdbg DEBUG, 'Sim: Response is command';
      }

#
# RSC fixme
#   my $dur = sprintf '%6.4f', time - $start;
   my $dur = sprintf '%6.4f', time - $self->_last_send_time;
   if ( length $self->_response < 8 )
      {
      $self->_errors_incr;
      logerr
         "<%s> Failed to receive complete response in %.2f seconds: %d",
         $self->_response, $dur, $self->_errors;
      $self->_flush if !$self->simulate;
      return 0;
      }
   else
      {
      logdbg DEBUG, "<%s> Received packet in %.2f seconds",
                    $self->_response, $dur;
      return 1;
      }
   }

#
# Special piece to try and deal with the noisy character which is sometimes
# seen.  We drop the first character and try to pick up a replacement, if
# appropriate
sub _off_one
   {
   my $self = shift;
   my $start = time;

   #
   # First off, make sure this is a one off candidate
   $self->_response =~ /^.(.).{5}./;

   if ( ! ( (defined $1)
          && ($1 eq $self->_check_chars_index( 0 ) )
          )
      )
      {
      return 0;
      }
   #
   # Clear out the response so we can start appending to it, and then
   # wait 'till we read an entire response, or time out
   if ( ! $self->simulate )
      {
      #
      # Drop the first character from the response
      $self->_response( substr $self->_response, 1 );
      #
      # Wait for the next one
      do
         {
         my( $count, $string ) = $self->_device->read( 1 );
         $self->_response( $self->_response . $string );
         $self->_noise_check;
         }
      while (  ( (time - $self->_last_send_time)
               < (RESPONSE_WAIT + $self->_extra_wait)
               )
            && (length $self->_response < 8)
            );
      }

   my $dur = sprintf '%6.4f', time - $self->_last_send_time;
   if ( length $self->_response < 8 )
      {
      $self->_errors_incr;
      logerr "<%s> Failed to receive 1 extra character in %d seconds: ",
             $self->_response, $dur, $self->_errors;
      $self->_flush if !$self->simulate;
      return 0;
      }
   else
      {
      $self->_extra_chars_incr;
      logdbg DEBUG, "<%s> Received 1 extra char in %d seconds",
                    $self->_response, $dur;
      return 1;
      }
   }

#
# Modify the command string to include the check characters
sub _prep_command
   {
   my $self = shift;

   #
   # First, go clear out the array, then make up 2 random chars from 0 to 9
   # and place them into the _check_chars array for later use
   $self->_check_chars_clear;
   my @chars = ( 0..9 );
   $self->_check_chars_push( @chars[ map{ rand @chars }( 1..2 ) ] );
   my( $c1, $c2 ) = $self->_check_chars;
   #
   # And now place those characters into positions 0 and 7 of the data to send
   my $command = $self->_command;
   $command =~ s/^.(.{6}).?/$c1$1$c2/;
   $command = $self->_command( $command );
   logdbg DEBUG, "<$command> Prepped with check bytes";
   }

#
# Checks the reshponse in self and returns 1 if the check characters
# match what was sent
sub _check_response
   {
   my $self = shift;

   #
   # Go and grab chars 0 and 7 to make sure they are the same as what we put
   # in (random characters)
   $self->_response =~ /^(.).{6}(.)/;

   if (  (defined $1)
      && (defined $2)
      && ($1 eq $self->_check_chars_index( 0 ))
      && ($2 eq $self->_check_chars_index( 1 ))
      )
      {
      logdbg DEBUG, "<%s> Check response passed", $self->_response;
      return 1;
      }
   else
      {
#      my $one = $1 || '';
#      my $two = $2 || '';
#      $self->_errors_incr;
#      logerr "<%s> Check response failed '%s' vs %s, '%s' vs %s",
#             $self->_response,
#             $one, $self->_check_chars_index(0),
#             $two, $self->_check_chars_index(1);
      return 0;
      }
   }

#
# Handles the sending and receiving of the command.  Note that the response
# will end up in the attribute "_response", and the command should be in
# "_command" already
sub _execute_command
   {
   my $self = shift;

   if ( length $self->_command != 8 )
      {
      logerr "<%s> Command not 8 characters long", $self->_command;
      return 0;
      }
   $self->_command_count_incr;
   $self->_prep_command;
   $self->_send;
   $self->_receive;
   my $check = $self->_check_response;
   #
   # Ok, here is an interesting one.  The stamp times out after 10 seconds,
   # and ends up sending a S........ string, one for each 10 second period.
   # But, the command sent appears to always(?) go through.  This little
   # bitty will clear out (as long as the first character is an S) every
   # 'timeout' string we receive
   while (  ($check == 0)
         && (substr( $self->_response, 0, 1 ) =~ /S/)
         )
      {
      #
      # We only need to flush 1 character as 8 were just received
      $self->_flush( 1 );
      $self->_timeouts_incr;
      logdbg WARN, "Received a stamp timeout: %d", $self->_timeouts;
      #
      # And try again
      $self->_receive;
      $check = $self->_check_response;
      }
   if ( !$check )
      {
      $self->_errors_incr;
      logerr 'Received a check response failure';
      return 0;
      }
   #
   # Make sure that the first response is also the command we sent
   if ( $self->_response ne $self->_command )
      {
      #
      # Last ditch effort to salvage the response.  Occationally, we get
      # suprious characters (seen during RA commands?), which this will
      # take care of
      $self->_off_one;
      if ( $self->_response ne $self->_command )
         {
         logerr 'Failed off one';
         return 0;
         }
      }
   $self->_receive( RESPONSE_CHAR );
   if ( !$self->_check_response )
      {
      $self->_errors_incr;
      logerr 'Received a check response failure';
      return 0;
      }
   return 1;
   }

sub _execute_command_retry
   {
   my $self = shift;

   my $retry = 0;

   my $status = $self->_execute_command;
   while ( $status == 0 )
      {
      last if ++$retry > $self->max_retry;
      $self->_retry_incr;
      logerr "Executing retry %d", $retry;
      $self->_flush;
      $status = $self->_execute_command;
      }
   $self->_extra_wait( 0 );
   return $status;
   }

sub _build_tag
   {
   my $buff;
   my $time = time;
   $buff = sprintf "%06.3f", $time % 60 + $time - floor $time;
   return "<$buff>";
   }

#
# This routine is responsible for returning an "ASCII'ed" array of characters,
# "$digits" characters long ("0" left padded), of the "$number" given
sub _map_number
   {
   my $self = shift;
   my( $digits, $number ) = @_;

   my $pattern = "%0?x";
   $pattern =~ s/\?/$digits/;
   my $new_number = sprintf $pattern, $number;

   my $result = '';

   while ( length $new_number )
      {
      $result = chr(ord( '0' ) + hex( chop $new_number )). $result;
      }
   confess
      "Constructed string ($result) not equal to digits requested ($digits)"
      if length $result != $digits;
   return $result;
   }

#
# This one does the reverse of _map_number - namely takes the "ASCII'ed"
# number and returning a real one
sub _map_ascii
   {
   my $self = shift;
   my( $string ) = @_;

   my $result = '';
   while ( length $string )
      {
      $result =   sprintf( "%x", (ord( chop $string ) - ord( '0' )) )
                . $result;
      }
   return hex $result;
   }

#
# Flushes the serial port
sub _flush
   {
   my $self = shift;
   my( $to_flush ) = @_;
   my $total = 0;
   my $flushed = '';

   #
   # Clear out any existing data - this will clear out 1 character, or
   # the amount asked to be flushed, if it's asked for
   if ( !$self->simulate )
      {
      my( $count, $char ) = $self->_device->read( $to_flush || 1 );
      $total += $count;
      $flushed .= $char;
      #
      # If no ammount was aksed to be flushed, go flush one at a time until
      # we time out
      if ( !defined $to_flush )
         {
         while ( $count )
            {
            ( $count, $char ) = $self->_device->read( 1 );
            $total += $count;
            $flushed .= $char;
            }
         }
      }
   if ( $total > 0 )
      {
      logdbg DEBUG, "flush - cleared $total characters";
      logdbg DEBUG, "flush - characters flushed: $flushed";
      }
   }

=item B<start>

=over

=item C<$td-E<gt>start>

The start method actually creates the serial port connection if not
under simulation.  Has to be called before any other methods are
used.

=item I<For Example>

=for example begin

 $td->start;

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub start
   {
   my $self = shift;

   if ( ! $self->simulate )
      {
      $self->_device( new Device::SerialPort( $self->port ) )
         or die 'Cannot open the serial port ' . $self->port . "\n";

      $self->_device->user_msg( 0 );
      $self->_device->baudrate( $self->baud );
      $self->_device->parity( 'none' );
      $self->_device->databits( 8 );
      $self->_device->stopbits( 1 );
      $self->_device->read_char_time( 4 );
      $self->_device->write_settings or die "Arg...";
      logtrc INFO, "Built connection with %s at %d",
                   $self->port, $self->baud;
      #
      # Clear out any existing data
      $self->_flush;
      }
   else
      {
      logtrc INFO, "Sim: Built connection with %s at %d",
                   $self->port, $self->baud;
      }

   $self->_started;
   }

=item B<Read ADC>

=over

=item C<(SCALAR, SCALAR) = $td-E<gt>read_adc( SCALAR )>

Reads the given adc channel, taking care of conversions and reading other adc
values if necessary.  Returns a status of 1 or 0 if the read completed
successfully or not and the value of the adc if successful.  Valid range is
from 0 to 31 inclusive.

=item I<For Example>

=for example begin

 ( $status, $adc ) = $td->read_adc( 0 );

=for example end

=item B<Simulation>

=over

=item C<(SCALAR, SCALAR) = $td-E<gt>read_adc( SCALAR, SCALAR, (SCALAR)?)>

When simulating, takes 1 or 2 extra SCALARs, which become the value(s) of the
ADCs which were requested.  Note that this is not the response string, but
the actual value(s) to be used.  If the adc read depends on another adc
value, the second value is required for the depended upon ADC.  The following
ADCs depend on another for their value, and will require the extra
SCALAR: 0, 1, 18, 19, 25, 26

=item I<For Example>

=for example begin

 ( $status, $adc ) = $td->read_adc( 19, 0xF000 );
 ( $status, $adc ) = $td->read_adc( 0, 0x4000, 0x1000 );

=for example end

=back

=back

=cut

sub read_adc
   {
   my $self = shift;
   my $adc = shift;
   my $sim_value;
   my $value;

   #
   # Check the value of the adc to read for validity
   if (   ($adc >= MAX_ADC)
       || ($adc < 0)
      )
      {
      logtrc ERROR, "read_adc - bad adc value $adc";
      return 0;
      }
   #
   # First off, build the adc read command - ".20xx00." where xx is
   # a value from 0x10 to 0x2F and .'s are the check digits
   $self->_command( sprintf ".%s%s00.", $self->_commands( 'ADC' ),
                                        $self->_map_number( 2, $adc + 0x10 ) );
   #
   # Then execute the command
   logdbg DEBUG, "read_adc - reading $adc";
   return 0 if ! $self->_execute_command_retry;
   if ( $self->simulate )
      {
      $self->_response( sprintf "..%s..", $self->_map_number( 4, shift ) );
      logdbg DEBUG, "<%s sim ADC read", $self->_response;
      }
   #
   # We have a valid response, so dig out the data
   $self->_response =~ /...(.{4})./;
   $value = $self->_map_ascii( $1 );

   $value -= 0x10000 if $value & 0x8000;
   $value =   ( $value / 3276.8 + $self->_adc_offset( $adc ) )
            * $self->_adc_scale( $adc );

   #
   # and this is the fun part.  If this value depends on another adc reading,
   # go and read that other adc value now...
   if ( $self->_adc_depend_exists( $adc ) )
      {
      logdbg INFO, 'read_adc - reading depended adc';
      my( $status, $d_value );
      my $dep_ref = $self->_adc_depend( $adc );
      if ( $self->simulate )
         {
         ($status, $d_value) = $self->read_adc( $dep_ref->[0], shift );
         }
      else
         {
         ($status, $d_value) = $self->read_adc( $dep_ref->[0] );
         }
      if ( !$status )
         {
         logdbg INFO, 'read_adc - bad read from depended adc';
         return $status;
         }
      logdbg INFO, "read_adc - ADC $adc = $value pre depend";
      $value -= $dep_ref->[ 1 ] + $d_value;
      }
   logdbg INFO, "read_adc - ADC $adc = $value";
   return 1, $value;
   }

=item B<Read Limits>

=over 

=item C<(SCALAR, HASH) = $td-E<gt>read_limits>

Reads the limit switches and returns both a SCALAR indicating the success or
failure of the operation (1 or 0) and a hash of the flags read.  Valid hash
elements currently are 'CAMERA_1', 'CAMERA_0', 'DEC_DRIVE' and 'RA'.

=item I<For Example>

=for example begin

 ($status, %limits) = $hash->read_limits;
 exists $limits{ 'CAMERA_O' };
 exists $limits{ 'CAMERA_1' };
 exists $limits{ 'RA' };
 exists $limits{ 'DEC_DRIVE' };

=for example end

=item B<Simulation>

=over

=item C<(SCALAR, HASH) = $td-E<gt>read_limits( SCALAR )>

When simulating, takes 1 extra SCALAR, which is decode to become the value of
the limits read.  Note that this is not the response string, but
the actual value to be used.

=item I<For Example>

=for example begin

 ($status, %limits)
    = $hash->read_limits( 0x82 );

=for example end

=back

=back

=cut

sub read_limits
   {
   my $self = shift;
   my %limits;
   #
   # First off, build the limit switch read command - ".60????." where ?'s
   # are don't care's and .'s are the check digits
   $self->_command( sprintf ".%2s????.", $self->_commands( 'SENSE' ) );
   #
   # Then execute the command
   logdbg DEBUG, 'read_limits - reading';
   return 0 if ! $self->_execute_command_retry;
   if ( $self->simulate )
      {
      $self->_response( sprintf ".....%s.", $self->_map_number( 2, shift ) );
      logdbg DEBUG, '<%s> Sim Limit read', $self->_response;
      }
   #
   # We have a valid response, so dig out the data
   for ( $self->_limits_keys )
      {
      my $ref = $self->_limits( $_ );
      my $byte = $self->_map_ascii( substr $self->_response, $ref->[0], 1 );
      $limits{ $_ } = ($byte & $ref->[1]) > 0 ? 1 : 0;
      }
   return 1, %limits;
   }

=item B<Write DAC>

=over

=item C<SCALAR = $td-E<gt>write_dac( SCALAR, SCALAR )>

Writes to the given dac the supplied value. The valid dac ranges are from
0 to 15 inclusive.  The value to write is given in volts.  Returns a status
of 1 or 0 if the write completed successfully or not.

=item I<For Example>

=for example begin

 $status = $td->write_dac( 15, 3.35 );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub write_dac
   {
   my $self = shift;
   my( $dac, $orig_value ) = @_;
   my $value;

   if (   ($dac >= MAX_DAC)
       || ($dac < 0)
      )
      {
      logtrc ERROR, "write_dac - bad dac value $dac";
      return 0;
      }
   #
   # First off, build the dac write command - ".1xxxyy." where xxx is the
   # encoded dac number, and yy is the encoded value to write
   $value = 140.0 - (($orig_value - 0.05) / .0974);
   if ( $value > 255 )
      {
      logtrc WARN, "write_dac - clipped $value to 255";
      $value = 255;
      }
   elsif ( $value < 0 )
      {
      logtrc WARN, "write_dac - clipped $value to 0";
      $value = 0;
      }
   $self->_command( sprintf ".%s%s0%s%s.",
                    $self->_commands( 'LOADADAC' ),
                    $self->_map_number( 1, $dac < 8 ? 0 : 1 ),
                    $self->_map_number( 1, ($dac % 8) * 2 ),
                    $self->_map_number( 2, $value ) 
                  );
   #
   # Then execute the command
   logdbg INFO, "write_dac - writing dac $dac with $orig_value";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=item B<Model Airplane Servo Operations>

=over

=item C<SCALAR = $td-E<gt>maservo( SCALAR, SCALAR )>

The first argument is the servo channel to move, the second is the position
requested.
The valid channels are 0 to 23, but only 22 (usually V camera shutter) and 14
(usually I camera shutter) are currently used.  'SERVO_0' may be used
in place of 22, and 'SERVO_1' in place of 14.
The position requested should be between 0 and 255 inclusive.
Causes the servo requested to move to the position requested.
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->maservo( 14, 65 );
 $status = $td->maservo( 'SERVO_0', 65 );
 $status = $td->maservo( 22, 255 );
 $status = $td->maservo( 'SERVO_1', 255 );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub maservo
   {
   my $self = shift;
   my( $channel, $position ) = @_;

   $channel = $self->_maservo_names( $channel )
      if $self->_maservo_names_exists( $channel );
   #
   # Do validity checking
   if (  ($channel < 0)
      || ($channel >= MAX_MASERVO_CHANNEL)
      )
      {
      logtrc ERROR, "maservo - bad channel: $channel";
      return 0;
      }
   if (   (!defined $position)
       || ($position < 0)
       || ($position >= MAX_MASERVO_POSITION)
      )
      {
      logtrc ERROR, "maservo - bad position: $channel";
      return 0;
      }
   #
   # Build the maservo write command - ".70xxyy." where xx is the
   # pulse channel and yy is the encoded positional value
   $self->_command( sprintf ".%s%s%s.",
                    $self->_commands( 'MASERVO' ),
                    $self->_pulses( $channel ),
                    $self->_map_number( 2, $position ),
                    );
   #
   # Need to wait some more for these sometimes???
   $self->_extra_wait( 2.5 );
   #
   # Then execute the command
   logdbg INFO, "maservo - writing $channel with position $position";

   if ( $self->ignore_maservo_errors )
      {
      $self->_execute_command;
      }
   else
      {
      return 0 if ! $self->_execute_command_retry;
      }
   return 1;
   }

=item B<Pulse Command>

=over

=item C<SCALAR = $td-E<gt>pulse( SCALAR )>

The argument is the channel to pulse. The valid channels are 0 to 23.
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->pulse( 22 );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub pulse
   {
   my $self = shift;
   my( $channel ) = @_;

   if (   ($channel < 0)
       || ($channel >= MAX_PULSE_CHANNEL)
      )
      {
      logtrc ERROR, "pulse - bad channel: $channel";
      return 0;
      }
   #
   # Build the pulse write command - ".30xx??." where xx is the
   # pulse channel and yy is the encoded positional value
   $self->_command( sprintf ".%s%s??.",
                    $self->_commands( 'PULSE' ),
                    $self->_pulses( $channel )
                    );
   #
   # Then execute the command
   logdbg INFO, "pulse - writing $channel";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=item B<Motor Setup>

=over

=item C<SCALAR = $td-E<gt>motor_setup( SCALAR, SCALAR )>

The first argument is the motor to set the direction for.  The second
argument is the direction to move the motor on the next move request.
Valid motors are 'FOCUS_0', 'FOCUS_1', 'DEC_DRIVE' and 'DEC_CLAMP'.
Valid directions are 'IN', -1, 'OUT', or 1.  'IN' and -1 will set up to move
the motor toward the limit sensor (or close the declination clamp motor),
while 'OUT' and 1 will set up to move the motor away from the limit sensor
(or open the delination clamp motor).
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->motor_setup( 'FOCUS_0', 'IN' );
 $status = $td->motor_setup( 'FOCUS_1', -1 );
 $status = $td->motor_setup( 'DEC_DRIVE', 'OUT' );
 $status = $td->motor_setup( 'DEC_CLAMP', 1 );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub motor_setup
   {
   my $self = shift;
   my( $motor, $direction ) = @_;

   if (  (!$self->_motors_exists( $motor ) )
      || (($self->_motors( $motor ))->[0] < 0)
      )
      {
      logtrc ERROR, "motor_setup - bad motor: $motor";
      return 0;
      }
   if ( !$self->_motor_directions_exists( $direction ) )
      {
      logtrc ERROR, "motor_seteup - bad direction: $direction";
      return 0;
      }
   # 
   # Tell the stamp which direction we're going to move
   $self->_command( sprintf ".%s%s%s00.",
                    $self->_commands( 'MOTOR_SETUP' ),
                    $self->_motor_directions( $direction ),
                    ($self->_motors( $motor ))->[0]
                  );
   #
   # Then execute the direction setup command
   logdbg INFO, "motor_setup - setting $motor to $direction";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=item B<Motor On and Off>

=over

=item C<SCALAR = $td-E<gt>motor_on( SCALAR )>

=item C<SCALAR = $td-E<gt>motor_off( SCALAR )>

The argument is the motor to turn on or off.
The valid motors are 'FOCUS_0', 'FOCUS_1', 'DEC_DRIVE' and 'DEC_CLAMP'.
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->motor_on( 'FOCUS_0' );
 $status = $td->motor_off( 'FOCUS_1' );
 $status = $td->motor_on( 'DEC_CLAMP' );
 $status = $td->motor_off( 'DEC_DRIVE' );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub motor_on
   {
   my $self = shift;
   my( $motor ) = @_;

   if (  (!$self->_motors_exists( $motor ))
      || (($self->_motors( $motor ))->[0] < 0)
      )
      {
      logtrc ERROR, "motor_on - bad motor: $motor";
      return 0;
      }
   #
   # Modify the motor state to indicate which motor will be on
   my $motors_on = $self->_motors_on;
   $motors_on |= ($self->_motors( $motor ))->[1];
   #
   # Set up the ON command with the proper motor selected
   $self->_command( sprintf ".%s59%s.",
                    $self->_commands( 'MOTOR_ON_OFF' ),
                    $self->_map_number( 2, ~$motors_on & 0xFF ),
                    );
   #
   # Then execute the on command
   logdbg INFO, "motor_on - turning on motor: $motor";
   return 0 if !$self->_execute_command_retry;
   #
   # and store which motors are on
   $self->_motors_on( $motors_on );
   return 1;
   }

sub motor_off
   {
   my $self = shift;
   my( $motor ) = @_;

   if (  (!$self->_motors_exists( $motor ))
      || (($self->_motors( $motor ))->[0] < 0)
      )
      {
      logtrc ERROR, "motor_off - bad motor: $motor";
      return 0;
      }
   #
   # Modify the motor state to indicate which motor will be now on
   my $motors_on = $self->_motors_on;
   $motors_on &= ~($self->_motors( $motor ))->[1];
   #
   # Set up the ON command with the proper motor selected
   $self->_command( sprintf ".%s59%s.",
                    $self->_commands( 'MOTOR_ON_OFF' ),
                    $self->_map_number( 2, ~$motors_on & 0xFF )
                    );
   #
   # Then execute the on command
   logdbg INFO, "motor_off - turning off motor: $motor";
   return 0 if !$self->_execute_command_retry;
   #
   # And change which motors are now on
   $self->_motors_on( $motors_on );
   return 1;
   }

=item B<Motor To Limit>

=over

=item C<SCALAR = $td-E<gt>motor_to_limit( SCALAR )>

The argument is the motor to move, with valid motors 'FOCUS_0', 'FOCUS_1',
'DEC_DRIVE'.
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->motor_to_limit( 'FOCUS_1' );
 $status = $td->motor_to_limit( 'FOCUS_0' );
 $status = $td->motor_to_limit( 'DEC_DRIVE' );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub motor_to_limit
   {
   my $self = shift;
   my( $motor ) = @_;

   if (  (!$self->_motors_exists( $motor ))
      || (($self->_motors( $motor ))->[3] == 0)
      )
      {
      logtrc ERROR, "motor_to_limit - bad motor: $motor";
      return 0;
      }
   #
   # Now, set up the command to move the motor to the limit
   $self->_command( sprintf ".%s%s%s.",
                    $self->_commands( 'MOTOR_MOVE_LIMIT' ),
                    $self->_map_number( 4, ($self->_motors( $motor ))->[3] )
                    );
   $self->_extra_wait( MAX_STEPS_TO_LIMIT * ($self->_motors( $motor ))->[ 4 ] );
   #
   # Then execute the move command
   logdbg INFO, "motor_to_limit - moving: $motor";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=item B<Motor Move>

=over

=item C<SCALAR = $td-E<gt>motor_move( SCALAR, SCALAR )>

The first argument is the motor to move, while the second is the number of
steps to move. The valid motors are 'FOCUS_0', 'FOCUS_1', 'DEC_DRIVE' and
'DEC_CLAMP'.  Steps are positive only, with the direction set with a call
to L<Motor Setup|"Motor Setup">.
Make sure the motor has been turn on using L<Motor On|"Motor On and Off">.
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->motor_move( 'FOCUS_0', 22 );
 $status = $td->motor_move( 'FOCUS_1', 20 );
 $status = $td->motor_move( 'DEC_DRIVE', 500 );
 $status = $td->motor_move( 'DEC_CLAMP', 200 );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub motor_move
   {
   my $self = shift;
   my( $motor, $steps ) = @_;

   if ( !$self->_motors_exists( $motor ) )
      {
      logtrc ERROR, "motor_move - bad motor: $motor";
      return 0;
      }
   if (  ($steps < 0)
      || ($steps >= MAX_STEPS)
      )
      {
      logtrc ERROR, "motor_move - bad number of steps: $steps";
      return 0;
      }
   #
   # Now, set up the command to move the motor the desired steps
   $self->_command( sprintf ".%s%s.",
                    $self->_commands( 'MOTOR_MOVE_STEPS' ),
                    $self->_map_number( 4, $steps )
                    );
   #
   # Add in the extra wait time
   $self->_extra_wait( $steps * ($self->_motors( $motor ))->[ 4 ] );
   #
   # Then execute the move command
   logdbg INFO, "motor_move - moving: $motor steps: $steps";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=item B<RA>

=over

=item C<SCALAR = $td-E<gt>motor_ra( SCALAR )>

The argument is one of 'OFF', 0, 'ON', 1, -1, 2, -2, 4, -4,
8, -8, 16, -16, 32, -32, 64, -64, 128, -128 which turn the RA motor off or on,
specify the speed and direction.  The numerical arguments specify the
multplier of siderial rate, with negative number moving opposite siderial.
'ON' and 1 move siderial rate to track stars.
Returns a status of 1 or 0 if the operation apparently completed successfully
or not.

=item I<For Example>

=for example begin

 $status = $td->motor_ra( 'ON' );
 $status = $td->motor_ra( 1 );
 $status = $td->motor_ra( 'OFF' );
 $status = $td->motor_ra( 0 );
 $status = $td->motor_ra( '128' );
 $status = $td->motor_ra( '-64' );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub motor_ra
   {
   my $self = shift;
   my( $cmd ) = @_;

   #
   # Check to make sure the command is valid
   if ( ! $self->_ra_cmds_exists( $cmd ) )
      {
      logtrc ERROR, "Bad ra command: $cmd";
      return 0;
      }
   #
   # Now, set up the command
   $self->_command( sprintf ".%s;;%s.",
                    $self->_commands( 'MOTOR_ON_OFF' ),
                    $self->_ra_cmds( $cmd ),
                    );
   #
   # Then execute the move command
   logdbg INFO, "motor RA: $cmd";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=item B<Load Register>

=over

=item C<SCALAR = $td-E<gt>load_register( SCALAR, SCALAR )>

Loads the specified register with the given value.  Currently supported
registers are 'MPX', 'POWER' and 'MOTOR'.

=item I<For Example>

=for example begin

 $status = $td->load_register( 'MPX', 0 );
 $status = $td->load_register( 'POWER', 0 );
 $status = $td->load_register( 'MOTOR', 0xFF );

=for example end

=item B<Simulation>

This method call does not change under simulation.

=back

=cut

sub load_register
   {
   my $self = shift;
   my( $register, $value ) = @_;

   if ( !$self->_registers_exists( $register ) )
      {
      logtrc ERROR, "load_register - invalid register: $register";
      return 0;
      }
   if (  ($value < 0)
      || ($value > 255)
      )
      {
      logtrc ERROR, "load_register - invalid value: $value";
      return 0;
      }
   #
   # Now, set up the command
   $self->_command( sprintf ".%s%s%s.",
                    $self->_commands( 'MOTOR_ON_OFF' ),
                    $self->_registers( $register ),
                    $self->_map_number( 2, $value )
                    );
   #
   # Then execute the move command
   logdbg INFO, "load_register - loading $register with $value";
   return 0 if ! $self->_execute_command_retry;
   return 1;
   }

=back

=head1 ATTRIBUTES

All attributes have three accessor methods, listed below.

=over

=item B<read>

=over

=item C<SCALAR = $obj-E<gt>x>

Returns the current value of attribute X

=back

=item B<write>

=over

=item C<SCALAR = $obj-E<gt>x( SCALAR )>

Sets and returns the value given for attribute x.  If the new value is undef,
the attributed will be unchanged, and the current value will be returned.

=back

=item B<clear>

=over

=item C<$obj-E<gt>clear_x>

Sets the value of attribute x to undef.

=back

=back

Currently, of the standard communication parameters, only the port address 
and baud rate are
configurable.  This saves work on my part and yours, as the remaining
parameters are set by the prom code.  If this changes, the code will be 
modified to add support for all modifiable parameters.

=over

=item B<port>

The serial port address which the device communicates over.  Typically will be
'/dev/ttyS0' or '/dev/ttyS1' for RedHat, Mandrake and possibly others.

=item B<baud>

The baud rate the device communicates.  Typically 9600 or 2400.

=item B<simulate>

Takes a value of 'true' or 'false', weather to simulate the serial connection
or use the real deal.  Note that many methods will take additional arguments
when simulating.

=back

=head1 AUTHOR

Robert Creager E<lt>Robert_Creager at LogicalChaos.orgE<gt>

=head1 SEE ALSO

=cut

1;
__END__
