//
// downfits: software to download CCD data from a MarkIV ISA card
// Copyright (C) 2001-2004  Robert S. Creager
//
// E-mail: Robert_Creager@LogicalChaos.org
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// $Name:  $
//
// $Log: downfits.c,v $
// Revision 1.1  2004/06/13 20:52:22  robert
// Add support for signed/unsigned fits files.  Move to popt for argument processing.
// Remove windows support.
//
// Revision 1.2  2004/06/02 02:47:11  robert
// *** empty log message ***
//
// Revision 1.1  2002/05/27 19:32:09  robert
// Add resetting of memory address lines
//
// Revision 1.0  2002/01/06 18:12:19  robc
// First CVS entry
//                                                                         

#include <sys/io.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <fitsio.h>
#include <longnam.h>
#include <popt.h>

void
outportb(
   unsigned short port,
   unsigned short value )
{
   return outb( value, port );
}

unsigned char
inportb(
   unsigned short port )
{
   return inb( port );
}

int
main(
   int argc,
   const char *argv[] )
{
   fitsfile *fptr[2];
   long naxis[2];
   char *ccd[2] = { "ccd0.fits", "ccd1.fits" };
   unsigned short *data[2];
   int status = 0;
   long x, y, offset;
   int num_ccds, ccd_count;

   num_ccds = 2;
   int signed_fits = 0;

   /*
    * Set up for popt
    */
   naxis[0] = 2064;
   naxis[1] = 2037;
   poptContext opt_con;
   int rc;
   struct poptOption options_table[] = {
      {"naxis1", 'n', POPT_ARG_INT | POPT_ARGFLAG_SHOW_DEFAULT, &naxis[0], 0,
       "Number of columns in the image",
       },
      {"naxis2", 'm', POPT_ARG_INT | POPT_ARGFLAG_SHOW_DEFAULT, &naxis[1], 0,
       "Number of rows in the image",
       },
      {"ccd0", 'c', POPT_ARG_STRING | POPT_ARGFLAG_SHOW_DEFAULT, &ccd[0], 0,
       "File name for data from CCD 0",
       },
      {"ccd1", 'd', POPT_ARG_STRING | POPT_ARGFLAG_SHOW_DEFAULT, &ccd[1], 0,
       "File name for data from CCD 1",
       },
      {"signed", 's', POPT_ARG_INT | POPT_ARGFLAG_SHOW_DEFAULT, &signed_fits,
       0,
       "When 1, fits values will range from -32767 to 32768. "
       "When 0, fits values will range from 0 to 65536.", "<0|1>"},
      POPT_AUTOHELP {0, 0, 0, 0, 0, 0, 0}
   };

   /*
    * And now do the argument parsing itself
    */
   opt_con = poptGetContext( NULL, argc, argv, options_table, 0 );

   if ( ( rc = poptGetNextOpt( opt_con ) ) < -1 )
      {
         fprintf( stderr, "%s: %s\n",
                  poptBadOption( opt_con, POPT_BADOPTION_NOALIAS ),
                  poptStrerror( rc ) );
         poptPrintHelp( opt_con, stderr, 0 );
         return 1;
      }

   /* Grant port permissions - need suid root privy */
   if ( ioperm( 0x300, 2, 1 ) )
      {
         printf
            ( "Cannot get read privledges for port 0x300 - is suid root set?\n" );
         perror( "ioperm" );
         exit( 1 );
      }

   /*
    * For each ccd, remove the associated file name, create the fits
    * file, create the image in each fits file, and allocate the memory
    * needed to store an entire image
    */
   for ( ccd_count = 0; ccd_count < num_ccds; ++ccd_count )
      {
         status = 0;
         remove( ccd[ccd_count] );

         fits_create_file( &fptr[ccd_count], ccd[ccd_count], &status );
         fits_report_error( stdout, status );

         /*
          * Create the file differently based on using signed or unsigned
          * data values
          */
         if ( signed_fits )
            {
               fits_create_img( fptr[ccd_count], SHORT_IMG, 2, naxis,
                                &status );
            }
         else
            {
               fits_create_img( fptr[ccd_count], USHORT_IMG, 2, naxis,
                                &status );
            }
         fits_report_error( stderr, status );

         fits_write_date( fptr[ccd_count], &status );
         fits_report_error( stdout, status );

         /* Allocate the needed memory for both images */
         data[ccd_count] = malloc( sizeof( unsigned short ) *
                                   naxis[0] * naxis[1] );
      }

   /*
    * Reset the memory board address register, should fix the problem where
    * data is shifted
    */
   // outportb( 0x300, 0x80 );
   // outportb( 0x300, 0x05 );
   // outportb( 0x300, 0x00 );

   /*
    * This resets the ISA card's address so we'll be reading from
    * the beginning, not from where we were last reading
    */
   outportb( 0x300, 0x1 );

   /* Now we go and read all the data for each CCD */
   for ( y = 0; y < naxis[1]; ++y )
      {
         offset = y * naxis[0];
         for ( x = offset; x < ( offset + naxis[0] ); ++x )
            {
               for ( ccd_count = 0; ccd_count < num_ccds; ++ccd_count )
                  {
                     data[ccd_count][x]
                        = ( ( unsigned short ) ( inportb( 0x301 ) ) << 8 );
                     data[ccd_count][x]
                        |= ( unsigned short ) ( inportb( 0x301 ) );
                     /* Adjust the value if we are using unsigned fits file */
                     if ( !signed_fits )
                        {
                           data[ccd_count][x] -= 32768;
                        }
                  }
            }
      }

   /*
    * Here we'll write out each image, and de-allocate the
    * memory used for the CCD read
    */
   for ( ccd_count = 0; ccd_count < num_ccds; ++ccd_count )
      {
         status = 0;
         /*
          * Again, do it a little different if we're using signed
          * or unsigned fits files
          */
         if ( signed_fits )
            {
               fits_write_img( fptr[ccd_count], TSHORT, 1,
                               naxis[0] * naxis[1], data[ccd_count],
                               &status );
            }
         else
            {
               fits_write_img( fptr[ccd_count], TUSHORT, 1,
                               naxis[0] * naxis[1], data[ccd_count],
                               &status );
            }
         free( data[ccd_count] );
      }

   /* Close out the final fits files */
   for ( ccd_count = 0; ccd_count < num_ccds; ++ccd_count )
      {
         status = 0;
         fits_close_file( fptr[ccd_count], &status );
         fits_report_error( stderr, status );
         /*
          * Since the executable is owned by root and suid, we want
          * to make the owner of the files whomever executed the program
          */
         if ( chown( ccd[ccd_count], getuid(  ), -1 ) )
            {
               perror( "chown" );
            }
      }

   /* Remove port access permissions */
   if ( ioperm( 0x300, 2, 0 ) )
      {
         printf
            ( "Cannot release read privledges for port 0x300 - is suid root set?\n" );
         perror( "ioperm" );
         exit( 1 );
      }
   exit( 0 );
}
